/*---------------------------------------------------------------------------+
| This file is part of Mumail, Copyright (c) 1992-1993 by
| Muhammad M. Saggaf. All rights reserved.
|
| See the file COPYING (1-COPYING) or the manual page mumail(1)
| for a full statement of rights and permissions.
+---------------------------------------------------------------------------*/

/*                               -*- Mode: C -*- 
 * MmFolder.c --- Folder handling
 * Author          : Muhammad M. Saggaf
 * Created On      : April 1993
 * Last Modified By: system admin
 * Last Modified On: Sun Jun 20 22:16:18 1993
 * Update Count    : 64
 * Status          : Mostly OK, needs some cleaning up
 */

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include "MuWin.h"
#include "MuGeneric.h"
#include "MmDecl.h"

#define PATHSEP_CHAR ('/')

#define PromptIffolderChanged(scn)\
  {if (scn->folder.changed) {\
	SimpleMessage(scn, "Changes were made to current folder. Discard them?");\
	if (BooleanPrompt(scn) == BP_NO) return;\
  }}

void
ParseFolderName(fullName, name, path)
	 String          fullName;
	 String          *name,
                     *path;
{
  String          sepPtr;

  fullName = ExpandTilda(fullName);
  if ((sepPtr = strrchr(fullName, PATHSEP_CHAR))) {
	*sepPtr = CNULL;
	*name = ++sepPtr;
	*path = fullName;
  }
  
  else {
	*path = FileInMailDir("");
	*(strend(*path)-1) = CNULL;
	*name = fullName;
  }
}

String
FileFullName(name, path)
	 String          name,
	                 path;
{
  return FmtString3("%s%c%s", path, PATHSEP_CHAR, name);
}

String
FolderFullName(scn)
	 SCREEN          *scn;
{
  return FileFullName(scn->folder.name, scn->folder.path);
}

String
FileBaseName(fullName)
	 String          fullName;
{
  String          tmpPtr = FmtString1("%s", fullName);
  String          name, path;

  ParseFolderName(tmpPtr, &name, &path);
  return name;
}

void
SetFolderName(scn, fullName)
	 SCREEN          *scn;
	 String          fullName;
{
  String          postTitle, name, path;

  ParseFolderName(fullName, &name, &path);

  if (scn->folder.name) XtFree(scn->folder.name);
  scn->folder.name = XtNewString(name);
  if (scn->folder.path) XtFree(scn->folder.path);
  scn->folder.path = XtNewString(path);

  postTitle = FmtString2("%s%s", scn->folder.name, scn->number ? 
						 FmtString1("(%d)", scn->number) : "");
  XtVaSetValues(scn->parentW, XtNtitle, FmtString1("Mumail: %s", postTitle),
				XtNiconName, postTitle, NULL);
}
  
int
OpenFolder(scn, folderName)
	 SCREEN          *scn;
	 String          folderName;
{
  static String          fullName;
  String                 name, path;

  /* This is here so that is the file chooser is not used, the folder
	 base name indicates it's in the mail directory */
  ParseFolderName(folderName, &name, &path);

  /* We want to make sure the name is still valid after the heavy use
	 of FmtString in GetLettersFromFolder() */
  if (fullName) XtFree(fullName);
  fullName = XtNewString(FileFullName(name, path));

  if (GetLettersFromFolder(scn, fullName) < 0) return -1;

  scn->folder.changed = False;
  SetFolderName(scn, fullName);
  
  SimpleMessage(scn, "Building table of contents...");
  ReMakeToc(scn);
  return 0;
}

/*---------------------------------------------------------------------------+
| Incorporate new mail routines.
+---------------------------------------------------------------------------*/

int
QueryMailBox(scn, mailBoxName)
	 SCREEN          *scn;
	 String          *mailBoxName;
{
  static String          mailBox;
  struct stat            statInfo;
  
  if (mailBox == NULL) mailBox = getenv("MAIL");
  if (mailBox == NULL) return QM_NO_MAILBOX_NAME;
  if (stat(mailBox, &statInfo) < 0) return QM_STAT_ERROR;

  *mailBoxName = mailBox;
  return (statInfo.st_size ? QM_MAILBOX_HAS_MAIL : QM_MAILBOX_EMPTY);
}

int
CheckMailBoxProc(scn, mailBox)
	 SCREEN          *scn;
	 String          *mailBox;
{
  int                    retStatus;

  switch (retStatus = QueryMailBox(scn, mailBox)) {
  case QM_NO_MAILBOX_NAME:
	SimpleMessage(scn, "Could not get mailbox name, aborting");
	return -1;
  case QM_STAT_ERROR:
	if (errno == ENOENT) SimpleMessage(scn, "No mailbox");
	else MessagePError(scn, "Mailbox error");
	return -1;
  case QM_MAILBOX_EMPTY:
	SimpleMessage(scn, "Mailbox is empty");
	return -1;
  case QM_MAILBOX_HAS_MAIL:
  default:
	break;
  }

  return retStatus;
}

void
CheckMailBoxCallback(widget, clientData, callData)
	 Widget             widget;
	 XtPointer          clientData;
	 XtPointer          callData;
{
  SCREEN          *scn = (SCREEN*)clientData;
  String          dummyS;

  if (CheckMailBoxProc(scn, &dummyS) == QM_MAILBOX_HAS_MAIL)
	SimpleMessage(scn, "Mailbox has mail");
}

int
IncorporateNewMailProc(scn)
	 SCREEN          *scn;
{
  String                 mailBox;
  FILE                   *mailBoxFP;
  String                 mailBoxBuf;
  int                    retStatus;

  if (CheckMailBoxProc(scn, &mailBox) != QM_MAILBOX_HAS_MAIL)
	return -1;

  if ((mailBoxFP = fopen(mailBox, "r")) == NULL) {
	MessagePError(scn, "Could not open Mailbox");
	return -1;
  }

  if ((mailBoxBuf = SimpleCopyStreamToBuffer(mailBoxFP)) == NULL) {
	MessagePError(scn, "Could not read Mailbox");
	fclose(mailBoxFP);
	return -1;
  }

  retStatus = AppendBufferToFile(FileInMailDir(INFOLDER_NAME), mailBoxBuf);

  fclose(mailBoxFP);
  XtFree(mailBoxBuf);

  if (retStatus < 0) return retStatus;

  if (truncate(mailBox, 0) < 0) {
	MessagePError(scn, "Could not empty mailbox");
	MuPError(FmtString1("Could not truncate file %s", mailBox));
  }

  return OpenFolder(scn, FileInMailDir(INFOLDER_NAME));
}

void
IncorporateNewMailCallback(widget, clientData, callData)
	 Widget             widget;
	 XtPointer          clientData;
	 XtPointer          callData;
{
  SCREEN          *scn = (SCREEN*)clientData;

  if (scn->folder.changed) {
	SimpleMessage(scn, "Changes were made to current folder. Discard them?");
	if (BooleanPrompt(scn) == BP_NO) return;
  }

  if (IncorporateNewMailProc(scn) >= 0) 
	SimpleMessage(scn, 
      FmtString1("New mail incorporated to folder %s, now the current folder", 
				 scn->folder.name));
}

/*---------------------------------------------------------------------------+
| Open folder routines.
+---------------------------------------------------------------------------*/

void
OpenFolderProc(scn, folderName)
	 SCREEN          *scn;
	 String          folderName;
{
  if (OpenFolder(scn, folderName) >= 0)
	SimpleMessage(scn, FmtString2("Folder %s loaded, has %d letters", 
								  scn->folder.name, scn->folder.numLetters));
}

void 
OpenFolderCallback(widget, clientData, callData)
	 Widget             widget;
	 XtPointer          clientData;
	 XtPointer          callData;
{
  SCREEN          *scn = (SCREEN*)clientData;
  String          fileName;

  PromptIffolderChanged(scn);
  if ((fileName = SelectFile(scn)) == NULL) return;
  OpenFolderProc(scn, fileName);
}

/*---------------------------------------------------------------------------+
| Re-open folder routine.
+---------------------------------------------------------------------------*/

void 
ReOpenFolderCallback(widget, clientData, callData)
	 Widget    widget;
	 XtPointer clientData;
	 XtPointer callData;
{
  SCREEN *scn = (SCREEN*)clientData;

  PromptIffolderChanged(scn);
  OpenFolderProc(scn, FolderFullName(scn));
}

/*---------------------------------------------------------------------------+
| Open folder in new screen routines.
+---------------------------------------------------------------------------*/

void 
OpenFolderInNextScreenCallback(widget, clientData, callData)
	 Widget    widget;
	 XtPointer clientData;
	 XtPointer callData;
{
  SCREEN          *scn = (SCREEN*)clientData, *nextScn;
  String          fileName;

  if ((fileName = SelectFile(scn)) == NULL) return;
  if ((nextScn = NextScreen(scn)) == NULL)
	nextScn = NewWindowProc(scn);

  OpenFolderProc(nextScn, fileName);
}

#if 0
void
UnlinkAndOpenFolderCallback(widget, clientData, callData)
	 Widget    widget;
	 XtPointer clientData;
	 XtPointer callData;
{
  _clientDataRec *clientDataRec = (_clientDataRec*)clientData;
  SCREEN *scn = (SCREEN*)clientDataRec->field[0];

  XtFree(scn->folder.name);
  scn->folder.name = XtNewString((String)clientDataRec->field[1]);

  if (unlink((String)clientDataRec->field[1]) < 0)
	Message(scn, 
	 FmtString("Error in opening folder %s, folder may not have been opened", 
			   scn->folder.name, "", ""), MSG_CLEAR_DEFAULT);
  else
	Message(scn, FmtString("Folder %s loaded", scn->folder.name, "", ""), 
			MSG_CLEAR_DEFAULT);
}

void
DoNewFolderCallback(widget, clientData, callData)
	 Widget    widget;
	 XtPointer clientData;
	 XtPointer callData;
{
  SCREEN *scn = (SCREEN*)clientData;
  static _clientDataRec clientDataRec;

  clientDataRec.field[0] = (XtPointer)scn;
  clientDataRec.field[1] = callData;

  if (access((String)callData, F_OK) < 0)
	{ExecOpenFolderCallback(widget, (XtPointer)&clientDataRec, callData);
	 return;}

  Message(scn, FmtString("Folder %s already exists. Overwrite it?", 
						 (String)callData, "", ""), MSG_CLEAR_DEFAULT);
  BooleanInputAndDispatch(scn, ExecNewFolderCallback, 
						  (XtPointer)&clientDataRec);
}
#endif

void 
NewFolderCallback(widget, clientData, callData)
	 Widget    widget;
	 XtPointer clientData;
	 XtPointer callData;
{
/*  SCREEN *scn = (SCREEN*)clientData;
  return;
  Message(scn, "Enter new folder name above", MSG_CLEAR_DEFAULT);
  DialogInputAndDispatch(scn, DoNewFolderCallback, (XtPointer)scn);*/
}

void 
SaveFolderCallback(widget, clientData, callData)
	 Widget             widget;
	 XtPointer          clientData;
	 XtPointer          callData;
{
  SCREEN          *scn = (SCREEN*)clientData;
  String          folderName;
  int             ret;
  
  ret = BackupFile(folderName = FolderFullName(scn));

  if (SaveFolder(scn, folderName) < 0) {
	SimpleMessage(scn, "Error in saving folder, folder not saved");
	return;
  }

  SimpleMessage(scn, FmtString1("Folder saved%s", ret < 0 ? 
								"" : ", old copy backed up"));
}

void
SaveFolderAs(scn, folderName)
	 SCREEN          *scn;
	 String          folderName;
{

  static String           fullName;
  String                  name, path, msg;
  int                     ret;

  /* This is here so that is the file chooser is not used, the folder
	 base name indicates it's in the mail directory */
  ParseFolderName(folderName, &name, &path);

  /* We want to make sure the name is still valid after the heavy use
	 of FmtString in GetLettersFromFolder() */
  if (fullName) XtFree(fullName);
  fullName = XtNewString(FileFullName(name, path));
  
  ret = BackupFile(fullName);

  if (SaveFolder(scn, fullName) < 0) {
	SimpleMessage(scn, "Error in saving folder, folder not saved");
	return;
  }

  SetFolderName(scn, fullName);
  msg = FmtString1("Folder saved as %s, now the current folder", 
				   scn->folder.name);
  if (ret >= 0)	strcat(msg, ", old copy backed up");
  SimpleMessage(scn, msg); 
}

void 
SaveFolderAsCallback(widget, clientData, callData)
	 Widget             widget;
	 XtPointer          clientData;
	 XtPointer          callData;
{
  SCREEN          *scn = (SCREEN*)clientData;
  String          fileName;

  if ((fileName = SelectFile(scn)) == NULL) return;
  SaveFolderAs(scn, fileName);
}

void
SaveFolderAsCompressedCallback(widget, clientData, callData)
	 Widget             widget;
	 XtPointer          clientData;
	 XtPointer          callData;
{
  SCREEN          *scn = (SCREEN*)clientData;
  String          newName = FmtString1("%s", FolderFullName(scn));

  if (FileIsPacked(newName)) PureFromPackedName(newName);
  SaveFolderAs(scn, FmtString1("%s.Z", newName));
}

void
SaveFolderAsGZippedCallback(widget, clientData, callData)
	 Widget             widget;
	 XtPointer          clientData;
	 XtPointer          callData;
{
  SCREEN          *scn = (SCREEN*)clientData;
  String          newName = FmtString1("%s", FolderFullName(scn));

  if (FileIsPacked(newName)) PureFromPackedName(newName);
  SaveFolderAs(scn, FmtString1("%s.gz", newName));
}

void
SaveFolderAsUnpackedCallback(widget, clientData, callData)
	 Widget             widget;
	 XtPointer          clientData;
	 XtPointer          callData;
{
  SCREEN          *scn = (SCREEN*)clientData;
  String          newName = FmtString1("%s", FolderFullName(scn));

  if (FileIsPacked(newName))
	*(strend(newName) - 2) = CNULL;

  SaveFolderAs(scn, newName);
}

void 
EditFolderCallback(widget, clientData, callData)
	 Widget             widget;
	 XtPointer          clientData;
	 XtPointer          callData;
{
  SCREEN          *scn = (SCREEN*)clientData;

  ErrorIfEditOrDialogMode(scn,);
  
  if (FileIsPacked(scn->folder.name)) {
	Message(scn, "Cannot edit a compressed folder", MSG_CLEAR_DEFAULT);
	return;
  }

  EditFileInBodyBox(scn, FolderFullName(scn));
  EditMode(scn, True);

  Message(scn, "For experts only, use caution", MSG_CLEAR_DEFAULT);
}

void 
DeleteFolderCallback(widget, clientData, callData)
	 Widget    widget;
	 XtPointer clientData;
	 XtPointer callData;
{
  SCREEN *scn = (SCREEN*)clientData;
  int    retStatus;

  SimpleMessage(scn, "Delete current folder? Are you really sure?");
  if (BooleanPrompt(scn) == BP_NO) return;

  retStatus = BackupFile(FolderFullName(scn));
  if (unlink(FolderFullName(scn)) < 0)
	MessagePError(scn, "Error in deleting folder, folder not deleted");
  else if (retStatus < 0)
	SimpleMessage(scn, "Folder deleted (but backup error)");
  else 
	SimpleMessage(scn, FmtString1("Folder deleted, backup saved as %s~", 
								  scn->folder.name));
}

int
CommitChanges(scn)
	 SCREEN *scn;
{
  /* We don't want to do anything if the folder doesn't have any letters
	 to begin with */
  if (scn->folder.numLetters == 0) return -1;

  CompactFolder(scn);
  ReMakeToc(scn);
  return 0;
}

void 
CommitChangesCallback(widget, clientData, callData)
	 Widget    widget;
	 XtPointer clientData;
	 XtPointer callData;
{
  SCREEN *scn = (SCREEN*)clientData;
  if (CommitChanges(scn) < 0)
	Message(scn, "No letters in folder, changes not commited", 
			MSG_CLEAR_DEFAULT);
  else Message(scn, "Changes to folder commited", MSG_CLEAR_DEFAULT);
}

/*---------------------------------------------------------------------------+
| Sort folder routine.
+---------------------------------------------------------------------------*/

void 
SortFolderCallback(widget, clientData, callData)
	 Widget    widget;
	 XtPointer clientData;
	 XtPointer callData;
{
  SCREEN          *scn = (SCREEN*)clientData;
  Letter          **letter = scn->letterList,
                  *tmpLetter = NULL;
  int             i, j;

  Message(scn, "Sorting folder...", MSG_CLEAR_DEFAULT);

  for (i = 0; i < scn->folder.numLetters; i++)
	for (j = i+1; j < scn->folder.numLetters; j++)
	  if (letter[i]->date.timeU > letter[j]->date.timeU) {
		tmpLetter = letter[i];
		letter[i] = letter[j];
		letter[j] = tmpLetter;
	  }

  ReMakeToc(scn);
  if (tmpLetter) scn->folder.changed = True;
  Message(scn, "Folder sorted", MSG_CLEAR_DEFAULT);
}

/*---------------------------------------------------------------------------+
| Search folder routine.
+---------------------------------------------------------------------------*/

void
SearchFolderCallback(widget, clientData, callData)
	 Widget             widget;
	 XtPointer          clientData;
	 XtPointer          callData;
{
  SCREEN          *scn = (SCREEN*)clientData;
  Letter          *letter;
  String          searchKey, letterBuffer;
  Boolean         found;
  int             i;

  SimpleMessage(scn, "Enter string to search letters for above");
  if ((searchKey = DialogPrompt(scn)) == NULL) return;

  AnimStart(scn);
  for (i = scn->folder.curLetN, found = False;
	   (letter = scn->letterList[i]) && !found; i++) {

	Anim(scn);
	letterBuffer = ConstructLetter(letter, LETTER_MAKE_WHOLE);
	if (CaseStrstr(letterBuffer, searchKey)) found = True;
		XtFree(letterBuffer);
  }

  AnimStop(scn);

  if (!found) {
	SimpleMessage(scn, "Search string not found");
	return;
  }

  scn->folder.curLetN = --i;
  RedisplayToc(scn);
  DisplayCurLetterPartialCallback(widget, clientData, callData);
  SimpleMessage(scn, 
	    FmtString("Found search string in letter #%d, now the current letter",
				  scn->folder.curLetN+1));

}

/*---------------------------------------------------------------------------+
| Folder info routine.
+---------------------------------------------------------------------------*/

void 
FolderInfoCallback(widget, clientData, callData)
	 Widget             widget;
	 XtPointer          clientData;
	 XtPointer          callData;
{
  SCREEN          *scn = (SCREEN*)clientData;
  String          curLetterMsg;
  char            message[4096];

  curLetterMsg = scn->folder.curLetterN < 0 ? "" :
	FmtString1(" (current is #%d)", scn->folder.curLetterN + 1);

  sprintf(message, "Current folder is %s, has %d letters%s%s", 
		  scn->folder.name, scn->folder.numLetters,
		  curLetterMsg, 
		  scn->folder.changed ? ", not saved" : "");
  SimpleMessage(scn, message);

  sprintf(message, "\n\t%s\n\n\t%s\n\t%s\n\n\t%s\n\t%s\n\t%s\n\n\t%s\n",
		  "---------- Folder Information ----------",
		  FmtString1("Base name:         %s", scn->folder.name),
		  FmtString1("Full name:         %s", FolderFullName(scn)),
		  FmtString1("# of letters:      %d", scn->folder.numLetters),
		  FmtString1("# tagged:          %d", scn->folder.numTagged),
 		  FmtString1("Current letter:    %s", 
					 scn->folder.curLetN < 0 ? "N/A" :
					 FmtString1("%d", scn->folder.curLetN + 1)),
		  FmtString1("Changed:           %s", scn->folder.changed ? 
					 "Yes" : "No"));
  DisplayBufferInBodyBox(scn, message);
}

void 
RedisplayTocCallback(widget, clientData, callData)
	 Widget    widget;
	 XtPointer clientData;
	 XtPointer callData;
{
  SCREEN          *scn = (SCREEN*)clientData;
  ReMakeToc(scn);
  Message(scn, "Table of contents reconstructed", MSG_CLEAR_DEFAULT);
}

String
SelectFile(scn)
	 SCREEN                  *scn;
{
#if USE_FILE_CHOOSER
  String                  FileChooser();
  String                  dir;
  static Boolean          firstTime = True;

  if (firstTime) {
	firstTime = False;
	dir = FileInMailDir("");
  }
  else dir = NULL;

  return FileChooser(scn->parentW, dir);

#else
  SimpleMessage(scn, "Enter name above (file chooser not compiled)");
  return DialogPrompt(scn);
#endif
}
