/*---------------------------------------------------------------------------+
| This file is part of Mumail, Copyright (c) 1992-1993 by
| Muhammad M. Saggaf. All rights reserved.
|
| See the file COPYING (1-COPYING) or the manual page mumail(1)
| for a full statement of rights and permissions.
+---------------------------------------------------------------------------*/

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/AsciiText.h>
#include <stdio.h>
#include <ctype.h>
#include "MuWin.h"
#include "MuGeneric.h"
#include "MmDecl.h"

String
GetTextSource(textW, startPos, endPos)
	 Widget                   textW;
	 XawTextPosition          startPos,
	                          endPos;
{
  static String          srcS;

  XtVaGetValues(textW, XtNstring, &srcS, NULL);
  
  if (endPos < strlen(srcS)) *(srcS+endPos+1) = CNULL;
  return srcS+startPos;

  /* Will be freed in the next call */
  
#if 0
 I don't know why this doesn't work, could e bug in Xaw
  Widget                 srcW = XawTextGetSource(textW);
  static XawTextBlock           textBlock;
  int                    len = endPos - startPos + 1,
                         totLen = 0;

  if (srcS) XtFree(srcS);
  /* add 1 for NULL-termination */
  srcS = XtMalloc(len + 1);
  
  strcpy(srcS, "");

  while (totLen < len) {
	totLen += XawTextSourceRead(srcW, startPos+totLen, &textBlock, len-totLen);
	strcat(strend(srcS), textBlock.ptr);
  }

  return srcS;
#endif
}

String
GetCurrentTextSelection(widget, maxLen)
	 Widget          widget;
	 int             maxLen;
{
  XawTextPosition startPos, endPos;

  XawTextGetSelectionPos(widget, &startPos, &endPos);
  if (startPos == endPos) return "";

  endPos = endPos-startPos < maxLen ? endPos : startPos+maxLen;
  return GetTextSource(widget, startPos, endPos);
}

XawTextPosition
FindStartOfCurrentLine(textW)
	 Widget          textW;
{
  return XawTextSourceScan(XawTextGetSource(textW), 
						   XawTextGetInsertionPoint(textW), XawstEOL, 
						   XawsdLeft, 1, False);
}

XawTextPosition
FindEndOfLine(textW, startPos)
	 Widget                   textW;
	 XawTextPosition          startPos;
{
  return XawTextSourceScan(XawTextGetSource(textW), startPos, XawstEOL, 
						   XawsdRight, 1, False);
}

XawTextPosition
FindEndOfCurrentLine(textW)
	 Widget          textW;
{
  return FindEndOfLine(textW, XawTextGetInsertionPoint(textW));
}

XawTextPosition
TextGetNthLineStartPos(textW, lineN)
	 Widget          textW;
	 int             lineN;
{
  return XawTextSourceScan(XawTextGetSource(textW), 0, XawstEOL, XawsdRight, 
						   lineN - 1, True);
  /* Or include = False and add 1 to return value  */
}

XawTextPosition
TextGetNthLineStartPosFromCurrentPos(textW, lineN)
	 Widget          textW;
	 int             lineN;
{
  return XawTextSourceScan(XawTextGetSource(textW), 
						   XawTextGetInsertionPoint(textW), XawstEOL, 
						   XawsdRight, lineN - 1, True);
  /* Or include = False and add 1 to return value  */
}

XawTextPosition
FindEndOfBuffer(textW)
	 Widget          textW;
{
  return XawTextSourceScan(XawTextGetSource(textW), 
						   XawTextGetInsertionPoint(textW), XawstAll, 
						   XawsdRight, 1, True);
}

void
HighlightCurrentLine(textW)
	 Widget          textW;
{
  XawTextSetSelection(textW, FindStartOfCurrentLine(textW), 
					  FindEndOfCurrentLine(textW));
}

void
HighlightToEndOFLine(widget)
	 Widget          widget;
{
  XawTextPosition  curPos;

  curPos = XawTextGetInsertionPoint(widget);
  XawTextSetSelection(widget, curPos, FindEndOfCurrentLine(widget));
}

void
ScrollTextToLine(textW, lineN)
	 Widget          textW;
	 int             lineN;
{
  String          textSource;
  XawTextPosition pos;

  /* not valid, see RedisplayToc */

  XawTextDisableRedisplay(textW);
  XtVaGetValues(textW, XtNstring, &textSource, NULL);
  pos = GetNthLinePos(textSource, lineN);
  printf("li= %d pos= %d\n", lineN, pos);
/*  XtVaSetValues(textW, XtNdisplayPosition, pos, NULL);*/
  XawTextSetSource(textW, XawTextGetSource(textW), pos);
}

void
CallAction(widget, action, param, numParam)
	 Widget    widget;
	 String    action;
	 String    *param;
	 Cardinal  numParam;
{
  XEvent event;
  Dimension width, height;
  Position x, y;

  XtVaGetValues(widget, XtNwidth, &width, XtNheight, &height, NULL);
  XtTranslateCoords(widget, (Position)width/2, (Position)height/2, &x, &y);
  event.type = ButtonPress;
  event.xbutton.x_root = x;
  event.xbutton.y_root = y;

  XtCallActionProc(widget, action, &event, param, numParam);
}

void
CallSimpleAction(widget, action)
	 Widget    widget;
	 String    action;
{
  CallAction(widget, action, NULL, 0);
}

void 
MoveToStartOfLine(textW)
	 Widget    textW;
{
  CallSimpleAction(textW, "beginning-of-line");
}

void 
MoveToEndOfLine(textW)
	 Widget    textW;
{
  CallSimpleAction(textW, "end-of-line");
}

void 
MoveToNextLine(textW)
	 Widget    textW;
{
  CallSimpleAction(textW, "next-line");
}

void 
MoveToPrevLine(textW)
	 Widget    textW;
{
  CallSimpleAction(textW, "previous-line");
}

void 
MoveToStartOfBuffer(textW)
	 Widget    textW;
{
  CallSimpleAction(textW, "beginning-of-file");
}

void 
MoveToEndOfBuffer(textW)
	 Widget    textW;
{
  CallSimpleAction(textW, "end-of-file");
}

void 
MoveToStartOfParagraph(textW)
	 Widget    textW;
{
  CallSimpleAction(textW, "backward-paragraph");
}


void 
MoveToEndOfPage(textW)
	 Widget    textW;
{
  Dimension                height;
  Widget                   textSinkW;
  XawTextPosition          displayPos;
  String                   arg[1];

  XtVaGetValues(textW, XtNheight, &height, XtNtextSink, &textSinkW, 
				XtNdisplayPosition, &displayPos, NULL);

  arg[0] = FmtString1("%d", XawTextSinkMaxLines(textSinkW, height) - 1);

  XawTextSetInsertionPoint(textW, displayPos);

  CallAction(textW, "multiply", arg, 1);
  CallSimpleAction(textW, "next-line");

  MoveToEndOfLine(textW);
}

int
InsertText(textW, textS)
	 Widget          textW;
	 String          textS;
{
  XawTextBlock             textBlock;
  XawTextPosition          loc;

  loc = XawTextGetInsertionPoint(textW);
  SetupTextBlock(textBlock, textS);

  if (XawTextReplace(textW, loc, loc, &textBlock) != XawEditDone)
	return -1;

  XawTextSetInsertionPoint(textW, loc + textBlock.length);
  return 0;
}

XawTextPosition
SearchForText(textW, textS, startPos, endPos, direction)
	 Widget                        textW;
	 String                        textS;
	 XawTextPosition               startPos,
                                   endPos;
	 XawTextScanDirection          direction;
{
  XawTextBlock     textBlock;
  XawTextPosition  findPos;

  SetupTextBlock(textBlock, textS);

  if ((findPos = XawTextSourceSearch(XawTextGetSource(textW), startPos, 
	  direction, &textBlock)) == XawTextSearchError || 
	  (endPos && findPos+textBlock.length > endPos))
	return XawTextSearchError;
  
  return findPos;
}

XawTextPosition
ReplaceTextByPos(textW, startPos, postEndPos, replS)
	 Widget                        textW;
	 XawTextPosition               startPos,
                                   postEndPos;
	 String                        replS;
{
  XawTextBlock             textBlock;

  SetupTextBlock(textBlock, replS);
  if (XawTextReplace(textW, startPos, postEndPos, &textBlock) != XawEditDone) 
	return XawEditError;
  
  return startPos;
}

XawTextPosition
ReplaceText(textW, origS, replS, startPos)
	 Widget                   textW;
	 String                   origS,
                              replS;
	 XawTextPosition          startPos;
{
  XawTextPosition          findPos;

  if ((findPos = SearchForText(textW, origS, startPos, 0, XawsdRight))
	  == XawTextSearchError) return XawEditError;

  /* Dont' subtract 1 from len since we want the the post-end position */
  return ReplaceTextByPos(textW, findPos, findPos + strlen(origS), replS);
}

int
InsertFile(textW, fileName)
	 Widget    textW;
	 String    fileName;
{
  FILE   *file;
  String  fileBuffer;
  int     retStatus;

  OpenStream(file, fileName, "r", -1);

  fileBuffer = SimpleCopyStreamToBuffer(file);
  fclose(file);

  retStatus = InsertText(textW, fileBuffer);
  
  XtFree(fileBuffer);
  return retStatus;
}

String
FormatParagraph(buffer, maxWidth)
	 String          buffer;
	 int             maxWidth;
{
  int             totalOffset,
                  curOffset,
                  lastSpacePos;
  char            c;
  String          parBuf, parBufPtr;

  parBufPtr = parBuf = XtMalloc(strlen(buffer) + 1);

/*  while (*buffer) {
    while ((c = *buffer) && isspace(c) && c != NEWLINE) buffer++;
    do *parBufPtr++ = c = *buffer++; while (c && c != NEWLINE);
  }*/

	/* This is short will
	 *  1) compress all multiple non-newline white space,
	 *  2) remove even non-multiple ones if at start of line/paragraph. */

  for (;(c = *buffer); buffer++) 
	if (!isspace(c) || c == NEWLINE ||
		(*(buffer+1) && !isspace(*(buffer+1)) && 
		 parBufPtr > parBuf && !isspace(*(parBufPtr-1))))
	  *parBufPtr++ = c == TAB ? SPACE : c;

  *parBufPtr = *buffer;         /* NULL-termination */

  for (lastSpacePos = curOffset = totalOffset = 0, c = parBuf[totalOffset];
	   c; totalOffset++, curOffset++) {

	c = parBuf[totalOffset];

	/* We want to keep all leading blank lines and preserve blank lines
	 * between paragraphs. This routine can accept a piece of text 
	 * spanning multiple paragraphs. */

	if (c == CNULL || (isspace(c) && ! (c == NEWLINE && (lastSpacePos == 0 ||
	   parBuf[totalOffset+1] == NEWLINE || parBuf[totalOffset+1] == CNULL)))) {
	  
	  if (c == '\n') parBuf[totalOffset] = SPACE;
	  
	  if (curOffset >= maxWidth) {
		parBuf[lastSpacePos] = NEWLINE;
		/* Leave the -1 alone. If removed, the first line in the 
		   paragraph will be longer than the rest */
		curOffset = totalOffset - lastSpacePos - 1;
	  }
	  lastSpacePos = totalOffset;

	}
  }	 /* FOR... */
	
  return parBuf;
}
