/*---------------------------------------------------------------------------+
| This file is part of Mumail, Copyright (c) 1992-1993 by
| Muhammad M. Saggaf. All rights reserved.
|
| See the file COPYING (1-COPYING) or the manual page mumail(1)
| for a full statement of rights and permissions.
+---------------------------------------------------------------------------*/

/*                               -*- Mode: C -*- 
 * MuGeneric.c --- Generic routines
 * Author          : Muhammad M. Saggaf
 * Created On      : April 1993
 * Last Modified By: system admin
 * Last Modified On: Sun Jun 20 16:19:06 1993
 * Update Count    : 16
 * Status          : Mostly OK, needs some cleaning up
 */

#include "config.h"
#include <X11/Intrinsic.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <signal.h>
#include <setjmp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include "MuGeneric.h"
#include "MmDecl.h"

/*---------------------------------------------------------------------------+
| Stream/buffer routines.
+---------------------------------------------------------------------------*/

int
OpenPFile(file, name, mode)
	 FILE            **file;
	 String          name,
	                 mode;
{
  FILE            *fp;
  String          pipeProg;          
  int             ret;

  if (FileIsPacked(name)) {
	signal(SIGPIPE, SIG_IGN);

	if (FileIsCompressed(name)) pipeProg = "compress";
	else pipeProg = "gzip";

	if (strcmp(mode, "r") == 0) {
	  if (access(name, R_OK) < 0) return -1;
	  fp = popen(FmtString("%s -dc %s 2> /dev/null", pipeProg, name),
				 mode); 
	}
	else if (strcmp(mode, "w") == 0)
	  fp = popen(FmtString("%s > %s 2> /dev/null", pipeProg, name), 
				 mode); 
	else return -2;

	ret = 1;
  }
  
  else {fp = fopen(name, mode); ret = 0;}
  
  *file = fp;
  if (fp) return ret;
  else return -1;
}
  
int
ClosePFile(file, name)
	 FILE            *file;
	 String          name;
{
  if (FileIsPacked(name)) return pclose(file);
  else return fclose(file);
}

#if 0
String
CopyStreamToBuffer(stream)
	 FILE *stream;
{
  int            fd = fileno(stream);
  Cardinal       fileSize;
  static char   *buffer;
  struct stat    statInfo;
  
  if (fstat(fd, &statInfo) < 0) {
	MuPError("Could not stat file");
	return NULL;
  }

  /* Add 1 for NULL-termination */
  buffer = XtMalloc((fileSize = (Cardinal)(statInfo.st_size)) + 1);

  if (read(fd, buffer, (int)fileSize) < 0) {
	MuPError("Could not read from file");
	XtFree(buffer);
	return NULL;
  }
  
  buffer[fileSize] = CNULL;
  return buffer;
}
#endif

/* This could be wastefull for small files */
#define BUFFER_SIZE (1<<19)		/* 512k */

String
CopyStreamToBuffer(stream, prependS)
	 FILE            *stream;
	 String          prependS;
{
  int                    fd = fileno(stream),
                         nBytes;
  Cardinal               totalAlloc, totalRead;
  static String          buffer;
  String                 bufPtr;

  /* Here we cannot assume that we know te size of the file (e.g. via 
	 stat(2)), since it could be a pipe */

  buffer = XtMalloc(totalAlloc = (totalRead = strlen(prependS)) + BUFFER_SIZE);
  strcpy(buffer, prependS);
  bufPtr = buffer + totalRead; 

  /* BUFSIZ is defiend in stdio.h */
  while ((nBytes = read(fd, bufPtr, BUFSIZ)) > 0) {
	totalRead += nBytes;
	if (totalAlloc - totalRead < BUFSIZ)
	  buffer = XtRealloc(buffer, totalAlloc += BUFFER_SIZE);
	bufPtr = buffer + totalRead;
  }

  if (nBytes < 0) {
	MuPError("Could not read from file");
	XtFree(buffer);
	return NULL;
  }

  /* Shrink (or grow by 1) the buffer to minimum required size */
  buffer = XtRealloc(buffer, totalRead+1);
  *(buffer + totalRead) = CNULL;

  return buffer;
}

String
SimpleCopyStreamToBuffer(stream)
	 FILE            *stream;
{
  return CopyStreamToBuffer(stream, "");
}

String
CopyFileToBuffer(fileName)
	 String          fileName;
{
  FILE            *fp;
  String          buffer;

  OpenStream(fp, fileName, "r", NULL);
  buffer = SimpleCopyStreamToBuffer(fp);
  fclose(fp);
  return buffer;
}

String
CopyPipeToBuffer(command)
	 String          command;
{
  FILE            *fp;
  String          buffer;

  signal(SIGPIPE, SIG_IGN);
  if ((fp = popen(command, "r")) == NULL) return NULL;
  buffer = SimpleCopyStreamToBuffer(fp);
  pclose(fp);
  return buffer;
}

int
AppendBufferToStream(stream, buffer)
	 FILE            *stream;
	 String          buffer;
{
  int           retStatus;

  if (fseek(stream, 0, SEEK_END) < 0) return -1;
  if ((retStatus = fputs( buffer, stream)) < 0)
	MuPError("Could not append buffer to stream");

  return retStatus;
}

int
AppendBufferToFile(fileName, buffer)
	 String          fileName;
	 String          buffer;
{
  FILE          *fp;
  int           retStatus;

  OpenStream(fp, fileName, "a", -1);
  retStatus = AppendBufferToStream(fp, buffer);
  fclose(fp);
  return retStatus;
}

int
AppendFileToStream(file, stream)
	 String          file;
	 FILE            *stream;
{
  String        buffer;
  int           retStatus;

  if ((buffer = CopyFileToBuffer(file)) == NULL) return -1;
  retStatus = AppendBufferToStream(stream, buffer);
  XtFree(buffer);
  return retStatus;
}

int
AppendFileToFile(file1, file2)
	 String file1;
	 String file2;
{
  String        buffer;
  int           retStatus;

  if ((buffer = CopyFileToBuffer(file1)) == NULL) return -1;
  retStatus =  AppendBufferToFile(file2, buffer);
  XtFree(buffer);
  return retStatus;
}

int
PipeBufferToCommand(command, buffer)
	 String          command;
	 String          buffer;
{
  FILE          *fp;
  int           retStatus;

  signal(SIGPIPE, SIG_IGN);
  if ((fp = popen(command, "w")) == NULL) return -1;
  if ((retStatus = fputs(buffer, fp)) < 0)
	MuPError("Could not write to pipe");
  
  pclose(fp);
  return retStatus;
}

#if 0
jmp_buf          pipeJmpEnv;

void
SigPipeHandler(sigNum)
	 int          sigNum;
{
  signal(SIGPIPE, SIG_DFL);
  longjmp(pipeJmpEnv, 1);
}

int
PipeBufferToCommand(command, buffer)
	 String          command;
	 String          buffer;
{
  FILE          *fp;
  int           retStatus;

  if ((fp = popen(command, "w")) == NULL) return -1;

  if (setjmp(pipeJmpEnv) != 0) {
	MuPError("Could not write to pipe");
	pclose(fp);
	return -1;
  }
  
  signal(SIGPIPE, SigPipeHandler);
  if ((retStatus = fputs(buffer, fp)) < 0)
	MuPError("Could not write to pipe");
  
  signal(SIGPIPE, SIG_DFL);
  pclose(fp);
  return retStatus;
}
#endif

String
PrependToBuffer(buffer, string)
	 String          buffer,
                     string;
{
  String          newBuf;

  newBuf = XtMalloc(strlen(buffer)+strlen(string)+1);
  strcat(strcpy(newBuf, string), buffer);

  XtFree(buffer);
  return newBuf;
}

/*---------------------------------------------------------------------------+
| Pipe routines.
+---------------------------------------------------------------------------*/

int
WriteToPipe(pd, data, size)
     int             *pd;
     String          data;
     int             size;
{
  int          retStatus;
  if ((retStatus = write(pd[1], data, size)) < 0)
    MuPError("Could not write to pipe");
  return retStatus;
}

int
ReadFromPipe(pd, data, size)
     int             *pd;
     String          data;
     int             size;
{
  int          retStatus;
  if ((retStatus = read(pd[0], data, size)) < 0)
    MuPError("Could not read from to pipe");
  return retStatus;
}

/*---------------------------------------------------------------------------+
| File routines.
+---------------------------------------------------------------------------*/

String
FileInHomeDir(fileName)
	 String fileName;
{
  static String          homeDir = NULL;

  if (homeDir == NULL) homeDir = getenv("HOME");
  if (homeDir == NULL) return fileName;
  return FmtString2("%s/%s", homeDir, fileName);
}

String
FileInMailDir(fileName)
	 String fileName;
{
  return FileInHomeDir(FmtString1("Mail/%s", fileName));
}

#define TILDA_CHAR '~'

String
ExpandTilda(fileName)
     String          fileName;
{
  static String          homeDir = NULL;
  String                 expandedName = FmtString0(""), 
                         curPtr = expandedName;
  int                    i;

  if (homeDir == NULL) homeDir = getenv("HOME");
  if (homeDir == NULL) return fileName;

  for (i = 0; fileName[i]; i++)
    if (fileName[i] == TILDA_CHAR) {
      strcpy(curPtr, homeDir);
      curPtr += strlen(homeDir);
    }
    else *curPtr++ = fileName[i];

  *curPtr = CNULL;
  return expandedName;
}

int MuSystem(cmd)
	 String cmd;
{
  int retStatus = system(cmd);
  if (WEXITSTATUS(retStatus) == 0) return 0;
  else return -1;
}

int
CopyFile(source, dest)
	 String          source;
	 String          dest;
{
  FILE            *sourceFP, *destFP;
  /* BUFSIZ is defined in stdio.h */
  char            buffer[BUFSIZ];
  size_t          numBytes;

  if ((sourceFP = fopen(source, "r")) == NULL) {
	MuPError(FmtString1("Could not open file %s for reading", source)); 
	return -1;
  }
  if ((destFP = fopen(dest, "w")) == NULL) {
	MuPError(FmtString1("Could not open file %s for writing", dest)); 
	fclose(sourceFP);
	return -1;
  }

  while ((numBytes = fread(buffer, sizeof(*buffer), BUFSIZ, sourceFP)))
	while (numBytes)
	  numBytes -= fwrite(buffer, sizeof(*buffer), numBytes, destFP);
  
  fclose(sourceFP);
  fclose(destFP);

  return 0;
}

int
BackupFile(fileName)
	 String fileName;
{
  /* If no old version of the file exits, no backup */
  if (access(fileName, F_OK) < 0) return -1;
  return CopyFile(fileName, FmtString1("%s~", fileName));
}

int
FileExists(name)
	 String          name;
{
  if (access(name, F_OK) == 0) return 1;
  else return 0;
}

int
IsRegularFile(name)
	 String          name;
{
  struct stat          statRec;

  if (stat(name, &statRec) < 0)
	{MuPError(FmtString1("Could not stat file %s", name)); return 0;}
  
  if (statRec.st_mode & S_IFREG) return 1;
  else return 0;
}

int
FileIsCompressed(name)
	 String          name;
{
  String          endPtr = strend(name);

  if (*(endPtr-2) == '.' && *(endPtr-1) == 'Z') return 1;
  else return 0;
}

int
FileIsGZipped(name)
	 String          name;
{
  String          endPtr = strend(name);

  if ((strlen(name) > 2 && strcmp(endPtr-2, ".z") == 0) ||
	  (strlen(name) > 3 && strcmp(endPtr-3, ".gz") == 0))
	return 1;
  else return 0;
}

#define PERIOD_CHAR '.'

String
PureFromPackedName(name)
	 String          name;
{
  String          endPtr = strend(name);
  
  if (!FileIsPacked(name)) return name;
  if (*(endPtr - 2) == PERIOD_CHAR) *(endPtr - 2) = CNULL;
  else *(endPtr - 3) = CNULL;
  return name;
}

/*---------------------------------------------------------------------------+
| truncate - hacked version that truncates only to zero length. For systems
|            that do not have truncate(2).
+---------------------------------------------------------------------------*/


#if !HAVE_TRUNCATE
int
truncate(path, length)
	 char          *path;
	 int           length;
{
  FILE          *fp;
  
  if (length) return -1;

  if ((fp = fopen(path, "w")))
	{fclose(fp); return 0;}
  
  return -1;
}
#endif

/*---------------------------------------------------------------------------+
| Misc. routines.
+---------------------------------------------------------------------------*/

void
FreeList(listArr)
     XtPointer       listArr[];
{
  int             i;

  for (i = 0; listArr[i]; i++)
    XtFree(listArr[i]);

  listArr[0] = NULL;
}

String
CurrentLocalTimeString()
{
  time_t curTime = time((time_t*)0);
  String timeS = ctime(&curTime);
  *(strend(timeS) - 1) = CNULL;
  return timeS;
}

/* max 1 hour */

int
ElapsedUTimeGreater(usec)
	 unsigned long          usec;
{
  static struct timeval          prevTime;
  struct timeval                 curTime;
  struct timezone                tzp;
  unsigned long                  diff;

  gettimeofday(&curTime, &tzp);
  
  diff = curTime.tv_sec - prevTime.tv_sec;

  /* To avoid overflow when multiplying by 10^6 */
  if (diff > (1 << 11)) diff = (1 << 11);

  diff *= 1000000L;
  diff += curTime.tv_usec - prevTime.tv_usec;

  if (diff < usec)
	return 0;

  prevTime = curTime;
  return 1;
}
  
int
ElapsedMTimeGreater(msec)
	 unsigned long          msec;
{
  return ElapsedUTimeGreater(msec*1000L);
}

/*---------------------------------------------------------------------------+
| String routines.
+---------------------------------------------------------------------------*/

String
FmtString(fmt, s1, s2, s3, s4, s5)
	 String          fmt, s1, s2, s3, s4, s5;
{

  /* Why so? So that calls to this function can nested 20 levels deep */
  static char buffer[20][1024];
  static int  curBuffer = 19;

  if (--curBuffer < 0) curBuffer = 19;
  sprintf(buffer[curBuffer], fmt, s1, s2, s3, s4, s5);
  return buffer[curBuffer];
}

String
strend(string)
	 char *string;
{
  return string + strlen(string);
}

/*---------------------------------------------------------------------------+
| FilterString - replaces unwanted characters with something else.
+---------------------------------------------------------------------------*/

String
FilterString(string, unwanted, replacement)
     String         string,
                    unwanted;
     char           replacement;
{
  int i, j;

  for (i = 0; string[i]; i++)
    for (j = 0; unwanted[j]; j++)
      if (string[i] == unwanted[j]) string[i] = replacement;

  return string;
}

/*---------------------------------------------------------------------------+
| StripStringB - removes leading white space from a string.
+---------------------------------------------------------------------------*/

String
StripStringB(string)
	 String          string;
{
  while (*string && isspace(*string)) string++;
  return string;
}

/*---------------------------------------------------------------------------+
| StripStringE - removes trailing white space from a string.
+---------------------------------------------------------------------------*/

String
StripStringE(string)
	 String          string;
{
  String          stringEnd;

  if (*string == CNULL) return string;

  stringEnd = strend(string) - 1;
  while (stringEnd > string && isspace(*stringEnd)) stringEnd--;

  if (isspace(*stringEnd)) *stringEnd = CNULL;
  else *++stringEnd = CNULL;

  return string;
}

/*---------------------------------------------------------------------------+
| StripStringA - removes leading and trailing white space from a string.
+---------------------------------------------------------------------------*/

String
StripStringA(string)
	 String          string;
{

  return StripStringE(StripStringB(string));
}

String
CleanSimpleCommentedBuffer(buffer)
	 String          buffer;
{
  return StripStringA(buffer);
}

String
CloneString(copy, orig)
	 String          *copy,
                     orig;
{
  if (*copy == orig) return *copy;

  if (*copy) XtFree(*copy);
  *copy = XtNewString(orig);
  return *copy;
}

/*
void
GetWord(lin, wrd)
     char           *lin,
                    *wrd;
{
  char           *ptr;
  int             cc = 0xff;
  int             quote = 0;

  ptr = wrd;
  if (lin) lptr = lin;

  *ptr = '\0';
  if (*lptr == '\0')
    return;

  while (isspace(*lptr))
    lptr++;

  wptr = lptr;

  if (*lptr == '\0')
    return;

  if (*lptr == '\"') {
    lptr++;
    quote = 1;
  }

  while (*lptr && (quote || !isspace(*lptr)) && (!quote || *lptr != '\"')) {
	
    if (*lptr == '^' && *(lptr + 1) && (!quote || *(lptr + 1) != '\"')) {
      lptr++;
      if (*lptr != '^')
		cc = 0x1f;
    }
    *ptr++ = *lptr & cc;
    lptr++;
    cc = 0xff;
  }

  if (*lptr)
    lptr++;;
  *ptr = '\0';
}


char*
NextWord(newLinePtr)
	 char *newLinePtr;
{
  static char nextWord[LRG_BUF], *linePtr;

  if (newLinePtr) linePtr = newLinePtr;
  GetWord(linePtr, nextWord);
  linePtr = lptr;
  return nextWord;
}
*/

#define QUOTE_CHAR ('\"')

String
NextWord(line)
	 String          line;
{
  static String          lastPos;
  String                 curPtr;
  Boolean                inQuote = False;

  if (line == NULL) line = lastPos;
  if (*(line = StripStringB(line)) == CNULL) return NULL;
  if (*line == QUOTE_CHAR) {line++; inQuote = True;}

  curPtr = line;

  while (*curPtr && 
		 (inQuote || !isspace(*curPtr)) && 
		 (!inQuote || *curPtr != QUOTE_CHAR)) curPtr++;

  if (*curPtr) {*curPtr = CNULL; curPtr++;}
  lastPos = curPtr;

  return line;
}

/*---------------------------------------------------------------------------+
| strstr - for systems that do not have it (or have a slow one).
+---------------------------------------------------------------------------*/

#define lower(c) (islower(c) ? c : tolower(c))

int
CaseStrSame(string1, string2)
	 String          string1,
                     string2;
{
  while (*string1 && *string2) 	
	if (lower(*string1++) != lower(*string2++)) return 0;

  if (*string1 || *string2) return 0;
  return 1;
}

int
CaseNStrSame(string1, string2, n)
	 String          string1,
                     string2;
	 int             n;
{
  register int          i;

  for (i = 0; *string1 && *string2 && i < n; i++, string1++, string2++)
	if (lower(*string1) != lower(*string2)) return 0;

  if (++i < n) return 0;
  return 1;
}

String
CaseStrstr(heyStack, needle)
     String          heyStack,
                     needle;
{
  register String          Sptr,
                           Tptr;
  int                      len = strlen(heyStack) - strlen(needle) + 1;

  if (*needle == CNULL) return NULL;

  for (; len > 0; len--, heyStack++) {
	if (lower(*heyStack) != lower(*needle))	continue;
	
	for (Sptr = heyStack, Tptr = needle; *Tptr != CNULL; Sptr++, Tptr++)
	  if (lower(*Sptr) != lower(*Tptr)) break;
	
	if (*Tptr == CNULL) return heyStack;
  }
  
  return NULL;
}

String
Strstr(heyStack, needle)
     String          heyStack,
                     needle;
{
#if HAVE_STRSTR && !HAVE_SLOW_STRSTR
  return strstr(heyStack, needle);

#else
  String          Sptr,
                  Tptr;
  int             len = strlen(heyStack) - strlen(needle) + 1;

  if (*needle == CNULL) return NULL;

  for (; len > 0; len--, heyStack++) {
	if (*heyStack != *needle)	continue;
	
	for (Sptr = heyStack, Tptr = needle; *Tptr != CNULL; Sptr++, Tptr++)
	  if (*Sptr != *Tptr) break;
	
	if (*Tptr == CNULL) return heyStack;
  }

  return NULL;
#endif
}

#if !HAVE_STRSTR
char*
strstr(heyStack, needle)
     char*           heyStack,
                     needle;
{
  return MuStrstr(heyStack, needle);
}
#endif

/*---------------------------------------------------------------------------+
| strerror - for systems that do not have it.
+---------------------------------------------------------------------------*/

#if !HAVE_STRERROR
char*
strerror(err)
     int err;
{
  extern char *sys_errlist[];

  return sys_errlist[err];
}
#endif

int
GetNthLinePos(buffer, n)
	 String          buffer;
	 int             n;
{
  int          line, i;

  for (i = 0, line = 1; line < n; i++, line++)
	while (buffer[i] != '\n') i++;

  return i;
}

/*---------------------------------------------------------------------------+
| Error messages and notices.
+---------------------------------------------------------------------------*/

void
MuError(msg)
     char           *msg;
{
  fprintf(stderr, ">> Error: %s.\n", msg);
  fflush(stderr);
}

#if 0
void
se_warning(msg)
     char           *msg;
{
  char            buf[REG_BUF];

  sprintf(buf, "\r>> Warning: %s.", msg);
  show(buf);
}

void
se_warningf(fmt, a, b, c)
     char           *fmt,
                    *a,
                    *b,
                    *c;
{
  char            buf[REG_BUF];

  sprintf(buf, fmt, a, b, c);
  se_warning(buf);
}
#endif

void
MuNotice(msg)
     char           *msg;
{
  fprintf(stderr, ">> Notice: %s.\n", msg);
  fflush(stderr);
}

void
MuPError(msg)
     char           *msg;
{
  fprintf(stderr, ">> Error: %s: %s.\n", msg, strerror(errno));
  fflush(stderr);
}


/*---------------------------------------------------------------------------+
| X11 generic routines.
+---------------------------------------------------------------------------*/

void
MuProcessEvent(widget)
	 Widget          widget;
{
  XtAppProcessEvent(XtWidgetToApplicationContext(widget), XtIMAll);
}
