/**********************************************************************
 *
 *	This client demonstrates a "receiver" client that can handle
 *	drops from some other client.  This application sets up two
 *	drop sites: a "green" box and a non-rectangular drop site that
 *	is painted initially as a "red" cross.
 *
 *	To demonstrate compatibility with OpenWindows 3.x clients, you
 *	can use the "source1" client shipped with OpenWindows and perform
 *	the drop on the "green" box.
 *
 *	To demonstrate compatibility with OSF/Motif 1.2 clients, drag
 *	colors from the DNDDemo client and drop them on the "red" cross.
 *
 *	In addition, the OI "sender" client can be used to drag both
 *	text to "green" box and colors to the "cross."
 *
 **********************************************************************
 */

#include <stdio.h>
#include <stdlib.h>
#include <OI/oi.H>


// the shape of our non-rectangular drop site
static OI_dnd_clip_rectangle	rect[] = {
	{ 15, 0, 20, 50 },
	{ 0, 15, 50, 20 }
};
#define NUM_RECTS	(sizeof(rect)/sizeof(OI_dnd_clip_rectangle))

/**********************************************************************
 *
 *	paint_rects
 *
 *		Repaint the "cross" drop site
 *
 **********************************************************************
 */
void 
paint_rects(
	OI_d_tech	*boxp,
	void		*,
	const XEvent	*)
{
	OI_connection	*conp;
	XRectangle	*xrects;
	int		i;

	xrects = (XRectangle *)malloc(NUM_RECTS * sizeof(XRectangle));
	for (i = 0; i < NUM_RECTS; i++) {
		xrects[i].x = rect[i].x;
		xrects[i].y = rect[i].y;
		xrects[i].width = (unsigned short)rect[i].width;
		xrects[i].height = (unsigned short)rect[i].height;
	}
	conp = boxp->connection();
	boxp->set_gc();
	XFillRectangles(conp->display(), boxp->X_window(), conp->gc(), xrects, NUM_RECTS);
	free((char*)xrects);
}

/**********************************************************************
 *
 *	drop_preview
 *
 *		The drop preview callback.  When we see an enter operation
 *		change the color of the border of the object to highlight it.
 *		When we see a leave, change it to the background so it 
 *		will seem to disappear
 *
 **********************************************************************
 */
void
drop_preview(
	OI_d_tech	*dtp,	// the object of interest
	void		*,	// argp
	OI_dnd_operation op,	// the drag and drop operation
	long,			// the X root coordinate of the pointer
	long)			// the Y root coordinate of the pointer
{
	switch (op) {
		case OI_dnd_enter:
			dtp->set_bdr_color("black");
			break;
		case OI_dnd_leave:
			dtp->set_bdr_pixel(dtp->bkg_pixel());
			break;
	}
}

/**********************************************************************
 *
 *	handle_drop
 *
 *		Something was dropped on the "green" box.  Look for the
 *		following data and place the results in the various
 *		entry_fields we created.
 *
 *			OI_dnd_file_name
 *			OI_dnd_string
 *			OI_dnd_length
 **********************************************************************
 */
void
handle_drop(
	OI_d_tech	*dtp,		// the object that caught the drop
	void		*,		// argp
	OI_dnd_operation op,		// the drag and drop operation
	long,				// the X root coordinate of the pointer
	long)				// the Y root coordinate of the pointer
{
	OI_app_window		*app;	// a pointer to the app_window
	char			*str;	// a temporary string variable
	int			*len;	// another temp variable
	char			buff[20];

	// find the enclosing app_window
	app = dtp->app_window();

	// try and get the OI_dnd_file_name data
	if ((str = (char*)dtp->get_drop_data(OI_dnd_file_name))) {
		// set the text to the appropriate entry_field
		((OI_entry_field*)app->subobject("filename"))->set_text(str);
		XFree(str);
	}

	// try and get the OI_dnd_string data
	if ((str = (char*)dtp->get_drop_data(OI_dnd_string))) {
		// set the text to the appropriate entry_field
		((OI_entry_field*)app->subobject("text"))->set_text(str);
		XFree(str);
	}

	// get either OI_dnd_host_name or OI_dnd_file_host_name
	if ((str = (char*)dtp->get_drop_data(OI_dnd_host_name)) == NULL)
		str = (char*)dtp->get_drop_data(OI_dnd_file_host_name);
	if (str) {
		// set the text to the appropriate entry_field
		((OI_entry_field*)app->subobject("host"))->set_text(str);
		XFree(str);
	}

	// now get the OI_dnd_length data
	if ((len = (int *)dtp->get_drop_data(OI_dnd_length))) {
		sprintf(buff, "%d", *len);
		XFree((char*)len);
		// set the text to the appropriate entry_field
		((OI_entry_field*)app->subobject("length"))->set_text(buff);
	}

	// signal the source object to delete its data if the operation was
	// a move. 
	if (op == OI_dnd_move)
		dtp->delete_drop_data();
}

/**********************************************************************
 *
 *	handle_color_drop
 *
 *		Something was dropped on the "cross" box.
 *		In this case, we're looking for a piece of data that 
 *		OI doesn't automatically fetch, "BACKGROUND"
 *
 **********************************************************************
 */
void
handle_color_drop(
	OI_d_tech	*dtp,		// the object that handled the drop
	void		*,		// argp
	OI_dnd_operation,		// the drag and drop operation
	long,				// the X root coordinate of the drop
	long)				// the Y root coordinate of the drop
{
	PIXEL			*color;

	color = (PIXEL *)dtp->get_drop_data("BACKGROUND");
	if (color) {
		dtp->set_fg_color(*color);
		XFree((char*)color);
		paint_rects(dtp, NULL, NULL);
	}
}

main (int argc, char **argv)
{
	OI_connection		*conp;
	OI_app_window		*app;
	OI_entry_field		*ef;
	OI_box			*box;
	int			row;

	if ((conp = OI_init(&argc, argv, "Receiver", "Receiver")) == NULL) {
		fprintf(stderr, "%s: OI_init failed\n", argv[0]);
		exit(1);
	}

	// create the top-level app_window
	app = oi_create_app_window("app_window", 200, 200, "Receiver");
	app->set_layout(OI_layout_row_aligned);

	row = 0;

	// create some entry fields we'll fill in when we handle a drop
	ef = oi_create_entry_field("filename", 40, "Filename:", " ", 200);
	ef->layout_associated_object(app, 0, row++, OI_ACTIVE);

	ef = oi_create_entry_field("text", 40, "Dropped Text:", " ", 200);
	ef->layout_associated_object(app, 0, row++, OI_ACTIVE);

	ef = oi_create_entry_field("host", 30, "From host:", " ", 200);
	ef->layout_associated_object(app, 0, row++, OI_ACTIVE);

	ef = oi_create_entry_field("length", 10, "Length:", " ", 200);
	ef->layout_associated_object(app, 0, row++, OI_ACTIVE);

	// create a green box to catch drops
	box = oi_create_box("box", 50, 50);
	box->set_bkg_color("green");

	// turn it into a drop site
	box->allow_drop();

	// setup the drop callback
	box->set_drop(handle_drop);

	// setup a drop preview callback
	// in the drop preview callback, we're simply going to change to color
	// of the border.  Here we'll give the box some border width and set the
	// initial color to the background so it won't show
	box->set_drop_preview(drop_preview);
	box->set_bvl_width(0);
	box->set_bdr_width(4);
	box->set_bdr_pixel(box->bkg_pixel());

	// put the box in the app_window
	box->layout_associated_object(app, 0, row, OI_ACTIVE);


	// Now we'll create a non-rectangular drop site
	box = oi_create_box("rectbox", 50, 50);

	// paint the cross in red
	box->set_fg_color("red");

	// setup an expose callback so we can repaint the cross
	box->set_expose(paint_rects);

	// setup the drop site with the rectangles to include
	box->allow_drop(rect, NUM_RECTS);

	// setup the drop handler callback
	box->set_drop(handle_color_drop);

	// setup a drop preview callback
	// in the drop preview callback, we're simply going to change to color
	// of the border.  Here we'll give the box some border width and set the
	// initial color to the background so it won't show
	box->set_drop_preview(drop_preview);
	box->set_bvl_width(0);
	box->set_bdr_width(4);
	box->set_bdr_pixel(box->bkg_pixel());

	// put the box in the app_window
	box->layout_associated_object(app, 1, row++, OI_ACTIVE);

	// put the app_window on the root
	app->set_associated_object(conp->root(), OI_DEF_LOC, OI_DEF_LOC, OI_ACTIVE);

	OI_begin_interaction();
	OI_fini();
}
