/***
 *** SVGATextMode, an SVGA textmode setting program
 *** (c) 1995 Koen Gadeyne (kmg@barco.be)
 ***
 *** Edited on 132x64 screen. Hence the LONG lines. For best viewing conditions, use this program to make itself more readable :-)
 ***/


/* If this is defined, on-the-fly screen resizing will not work, but it will at least compile and run on older kernels
 * It will also throw out kernel version checking, and "stty"-type resizing.
 * In short: anything to do with resizing is out.
 */

/*
#define NO_VT_RESIZE
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <sys/types.h>
#include <asm/io.h>

#ifndef NO_VT_RESIZE
  #include <linux/vt.h>
  #include <sys/utsname.h>
#endif

#include <sys/ioctl.h>
#include <termios.h>
#include <fcntl.h>

#include "vga_prg.h"
#include "setclock.h"
#include "configfile.h"
#include "messages.h"

#ifndef TRUE
#define TRUE (1)
#endif
#ifndef FALSE
#define FALSE (-1)
#endif


#define MAX_ATTRIBS         4
#define ATTRIB_LEN          10
#define MAX_CLOCKDEVIATION  3.0
#define MAX_TERMINALS       32   /* if you have more terminals to resize than that, increase this numner */ 

#define DEFAULT_FONTPROGPATH  "/usr/bin/setfont"
#define DEFAULT_FONTPATH      "/usr/lib/kbd/consolefonts"

/*
 * global variables
 */

char *CommandName;
int debug_messages=FALSE;
int vgaIOBase;

void usage()
{
     PMESSAGE(("version %s. (c) 1995 Koen Gadeyne (kmg@barco.be)\n Usage: %s [options] textmodelabel\n
     Options: -n  Don't program VGA hardware (parse config file and report mode parameters)
              -d  print debugging information
              -h  print usage information
              -r  don't run ResetProg
              -f  don't run FontProg
              -c  don't reprogram pixel clock (for debugging)\n
     Textmodelabel: an existing label from the config file %s\n",VERSION, CommandName, CONFIGFILE))
}

/****************************************************************************************************************************/
int main (int argc, char* argv[])
{
  char* req_label;
  char *arg_str, *prev_arg_str;
  char tempstr[1024]="";
  char extclockpath[1024]="";
  char resetprogpath[1024]="";
  char fontprogpath[1024]=DEFAULT_FONTPROGPATH;
  char fontpath[1024]=DEFAULT_FONTPATH;
  char *font_table[2][32];
  int resetprog=FALSE;
  int extclock=FALSE;
  int extclockindex=0;
  int clockchip=-1;  /* < 0 means NO clock chip */
  const char *str_chipsets[NUM_CHIPSETS] = CHIPSET_STRINGS;
  const char *str_options[NUM_OPTIONS] = OPTION_STRINGS;
  const int str_allowedopts[NUM_CHIPSETS] = OPTIONS_ALLOWED;
  const char *str_attributes[NUM_ATTRIBS] = ATTRIB_STRINGS;
  const char *str_clkchips[NUM_CLKCHIPS] = CLKCHIP_STRINGS;

  int chipset = FALSE;
#ifdef ALLOW_CLOCKDIV2
  float clocks[MAX_CLOCKS*2];
#else
  float clocks[MAX_CLOCKS];
#endif
  int num_clocks=0;
  int optionmask=0;
  int attribval=-1;
  int program_hardware=TRUE;
  char c;
    
  float clockfreq, realclock, horfreq, vertfreq, tempf;
  int activepix, start_hsync, stop_hsync, totalpix;
  int activelines, start_vsync, stop_vsync, totallines;
  char attrib[MAX_ATTRIBS][ATTRIB_LEN];
  
  int textlines;
  int blanklines;
  int charsperline;
  int cursor_start=0, cursor_end=15;

  struct winsize my_winsize;
  int dev;

#ifndef NO_VT_RESIZE
  struct vt_sizes my_vt_size;      /* this struct passes the new screen size on to the kernel */
  char* tmpmem;
  char termstrings[MAX_TERMINALS][64];
  int terminals=0;
  struct utsname my_utsname;
  int kversion, ver, rel, pl;
  int sresize=TRUE;         /* will be set when screen has been resized and vice-versa */
#endif
  
  int i=0, j=0, x=0, y=0, result=0, opt=0;

  int font_height=16, font_width=8;      /* height/width of character cell : default is 8x16 */
  int h_polarity = -1, v_polarity = 1; /* default sync polarities: May not be OK */
  
  int run_resetprog=TRUE, run_fontprog=TRUE, run_pixclock=TRUE;
  int change_cursor=FALSE;
  
  FILE *param_file;
    

 /*
  * command-line argument parsing
  */

  CommandName = argv[0];

  while ((c = getopt (argc, argv, "ndhrfc")) != EOF)
    switch (c)
    {
      case 'n': program_hardware=FALSE;
                break;
      case 'd': debug_messages=TRUE;
                break;
      case 'h': usage();
                exit(0);
                break;
      case 'r': run_resetprog=FALSE;
                break;
      case 'f': run_fontprog=FALSE;
                break;
      case 'c': run_pixclock=FALSE;
                break;
      case '?': usage();
                PERROR(("Bad option '%s'\n",argv[optind-1]));
                exit(-1);
                break;
      default: PERROR(("getopt returned unknown token '%c'.",c));
    }
  if (optind < argc)
     req_label = argv[optind];
  else
     {
       usage();
       PERROR(("No textmode label on commandline."));
     }

 /*
  * open parameter file
  */
  param_file = open_param_file(CONFIGFILE);

 /*
  * get chipset definition 
  */
  sscanf(findlabel(param_file, "ChipSet", LABEL_REQUIRED+LABEL_FROMSTART), "%*s %s", tempstr);
  chipset = findoption(tempstr, str_chipsets, NUM_CHIPSETS, "chip set");
  
 /*
  * Option parsing. Not all chips support all options
  */
  optionmask = 0;
  if ((arg_str = strtok(findlabel(param_file, "Option", LABEL_OPTIONAL+LABEL_FROMSTART)," ")) != NULL)
  {
    do
    { 
      if ((arg_str=strtok(NULL," ")) == NULL) PERROR(("Empty 'Option' line in config file '%s'.", CONFIGFILE));
      opt = findoption(arg_str, str_options, NUM_OPTIONS, "Option");
      if (opt >= NUM_OPTIONS) PERROR(("Illegal Option line in config file"));
      /* check if option allowed for selected chip set */
      if ( !(str_allowedopts[chipset] & (1<<opt)))
        PERROR(("Option '%s' not allowed for chipset '%s'.", str_options[opt], str_chipsets[chipset]));
      optionmask |= (1<<opt);  /* make optionvalues bitmask */
    } while ((arg_str = strtok(findlabel(param_file, "Option", LABEL_OPTIONAL+LABEL_FROMCURRENT)," ")) != NULL);
  }
  PDEBUG(("Options selected (bit mask): 0x%x", optionmask));
  
 /*
  * get pixel clocks (multiple lines possible)
  */

  /* first see if external clock program is used */
  extclock = GetPathOption(param_file, "ClockProg", LABEL_OPTIONAL+LABEL_FROMSTART, extclockpath);
  if (extclock==TRUE)
  {
    /* get clock index --- this MUST change in the future: is a bad hack: assumes GetPathOption uses strtok() ! */
    if ((arg_str=strtok(NULL," ")) != NULL) extclockindex = getbyte(arg_str, "external clock index", 0, 3);
    PDEBUG(("External clock program will use clock index %d", extclockindex));
  }
  else  /* check for special clock chip */
  {
    clockchip=-1;
    if (strtok(findlabel(param_file, "ClockChip", LABEL_OPTIONAL+LABEL_FROMSTART)," ") != NULL)
    {
      if (chipset != CS_S3) PERROR(("ClockChip selection is currently  only allowed for S3 chipsets"));
      if ((arg_str=strtok(NULL," ")) == NULL) PERROR(("'ClockChip': clock chip type not defined"));
      clockchip = findoption(arg_str, str_clkchips, NUM_CLKCHIPS, "Clock Chip");
      PDEBUG(("Using clock chip '%s'", str_clkchips[clockchip]));
    }
    else  /* get clocks lines */
    {
      switch (chipset)
      {
        case CS_CIRRUS: break;   /* no clocks for Cirrus */
        default:
            {
              arg_str = strtok(findlabel(param_file, "Clocks", LABEL_REQUIRED+LABEL_FROMSTART)," ");
              num_clocks = 0;
              do
              {
                while ((arg_str=strtok(NULL," ")) != NULL)
                  clocks[num_clocks++] = getfloat(arg_str, "'Clocks' definition: clock frequency in configuration file", MIN_CLOCK, MAX_CLOCK);
                if ((num_clocks <4) || (num_clocks >MAX_CLOCKS))
                   PWARNING(("Unusual number of clocks (%d) in config file '%s'",num_clocks,CONFIGFILE));
                PDEBUG(("Clocks: found %d clocks (until now)",num_clocks));
              }
              while ((arg_str=strtok(findlabel(param_file, "Clocks", LABEL_OPTIONAL+LABEL_FROMCURRENT)," ")) != NULL);
              for (i=0; i<num_clocks; i++) sprintf(&tempstr[i*7], "%7.2f ", clocks[i]);
              PDEBUG(("Clocks: found %d clocks: %s",num_clocks, tempstr));
            }
      }
    }
  }

 /*
  * check for correct amount of clocks, if possible for the specified chip set. This could be incorrect...
  */
  if ((chipset == CS_VGA) && (num_clocks > 4))
    PERROR(("Generic VGA chipsets can have no more that 4 clocks (and not %d).", num_clocks));
  if ((chipset == CS_PVGA1) && (num_clocks != 8) && (num_clocks != 4))
    PERROR(("PVGA1 chipset must have 4 or 8 clocks in 'clocks' line"));
  if (((chipset == CS_WDC90C0X) || (chipset == CS_WDC90C1X) || (chipset == CS_WDC90C2X)) && (num_clocks != 9))
    PERROR(("WDC90C0X, 1X and 2X chipsets have 8 pixel clocks PLUS the MClock."));
  if ((chipset == CS_WDC90C3X) && (num_clocks != 17))
    PERROR(("WDC90C3X chipsets have 16 pixel clocks PLUS the MClock."));
  if (((chipset==CS_ATI) || (chipset==CS_ATIMACH32)) && (num_clocks != 64))
    PWARNING(("ATI chipsets (VGA Wonder V5 and above) normally have 64 clocks."));
         

 /*
  * find last occurence of requested text mode line: This will allow user-patched mode lines at end of TextConfig file
  * to get preference over the ones above it, which normally are the default ones (as suggested by Kenneth Albanowski).
  */
  arg_str = findlabel(param_file, req_label, LABEL_REQUIRED+LABEL_FROMSTART+LABEL_LAST);
  result = sscanf(arg_str, "%*s %f  %d %d %d %d  %d %d %d %d   %s %s %s %s\n",
              &clockfreq,
              &activepix, &start_hsync, &stop_hsync, &totalpix,
              &activelines, &start_vsync, &stop_vsync, &totallines,
              attrib[0], attrib[1], attrib[2], attrib[3]);
  if (result<10)
     PERROR(("Badly formed textmode config string at label '%s' in config file '%s'",req_label,CONFIGFILE));
     
 /*
  * Do some SEVERE error checking on the text mode string timings!
  * the ranges are somewhat randomly chosen. Need to study REAL hardware limits for this...
  */
  
  check_int_range(activepix, 16, 4096, "active pixels");
  check_int_range(start_hsync, activepix, 4096, "start of H-sync");
  check_int_range(stop_hsync, start_hsync+1, start_hsync+(32*8), "end of H-sync");
  check_int_range(totalpix, 16, 4096, "total pixels");

  check_int_range(activelines, 16, 4096, "active lines");
  check_int_range(start_vsync, activelines, 4096, "start of V-sync");
  check_int_range(stop_vsync, start_vsync+1, start_vsync+16, "end of V-sync");
  check_int_range(totallines, 16, 4096, "total lines");
  
 /*
  * Auto sync polarity
  */
  if (activelines < 400)      { h_polarity = 1 ; v_polarity = -1; }
  else if (activelines < 480) { h_polarity = -1 ; v_polarity = 1; }
  else if (activelines < 768) { h_polarity = -1 ; v_polarity = -1; }
  else { h_polarity = 1 ; v_polarity = 1; }

 /*
  *  parse any attribute strings
  */
  for (i=0; i<(result-9); i++)   /* supposing "result" is still the result from the sscanf ... */
  {
     attribval = findoption(attrib[i], str_attributes, NUM_ATTRIBS, "attribute");
     switch(attribval)
     {
        case ATTR_FONT: i++;  /* get next attribute, which SHOULD be the font SIZE */
                        if (i >= (result-9))
                           PERROR(("'font' attribute must be followed by a font size (e.g. 8x16)"));
                        ParseFontXY(attrib[i], &font_width, &font_height);
                        break;
        case ATTR_PHSYNC: h_polarity = 1;
                          break;
        case ATTR_NHSYNC: h_polarity = -1;
                          break;
        case ATTR_PVSYNC: v_polarity = 1;
                          break;
        case ATTR_NVSYNC: v_polarity = -1;
                          break;
        default: PERROR(("Unknown attribute '%s' in config line '%s'.", attrib[i], req_label));
     }
  }
     
  PDEBUG(("Font size will be %dx%d", font_width, font_height));
  PDEBUG(("sync polarities: Hsync: %c, Vsync: %c", (h_polarity<0) ? '-' : '+', (v_polarity<0) ? '-' : '+'));
  
#ifndef NO_VT_RESIZE
 /*
  * See what terminals will need resizing, if any are given.
  */
  terminals=0;
  if (strtok(findlabel(param_file, "Terminals", LABEL_OPTIONAL+LABEL_FROMSTART)," ") != NULL)
  {
    while ((arg_str=strtok(NULL," ")) != NULL)
    {
      strcpy(termstrings[terminals], "/dev/");
      strcat(termstrings[terminals++], arg_str);
      if (terminals >= MAX_TERMINALS) PERROR(("Too many terminals defined on 'Terminals' line (max=%d)", MAX_TERMINALS));
    }
  PDEBUG(("Terminals: found %d terminal strings", terminals));
  }
  else
  {
    PWARNING(("No 'Terminals' line in config file."));
    PWARNING(("  Screen resizing will NOT be checked for kernel incompatibility."));
  }

 /*
  * Check on kernel version. Older than 1.1.54 will result in warning, and not allowing user to resize the screen.
  */
  if (uname(&my_utsname))
  {
    perror("uname");
    PERROR(("Could not get kernel version number."));
  }
  PDEBUG(("Running on kernel version: '%s'", my_utsname.release)); 
  sscanf(my_utsname.release, "%d.%d.%d", &ver, &rel, &pl);
  kversion = ver*1000000+rel*1000+pl;

 /*
  * get reset-program (will be called at end of SVGATextMode mode switch)
  */

  resetprog = GetPathOption(param_file, "ResetProg", LABEL_OPTIONAL+LABEL_FROMSTART, resetprogpath);

#endif

 /*
  * get font program path and font path. If not defined, default path will be used.
  * also get the default font table.
  */

  if (optionmask & OPT_LOADFONT)
  {
    if (GetPathOption(param_file, "FontProg", LABEL_OPTIONAL+LABEL_FROMSTART, fontprogpath) == FALSE)
      PDEBUG(("Using default FontProg path '%s'", fontprogpath));
    if (GetPathOption(param_file, "FontPath", LABEL_OPTIONAL+LABEL_FROMSTART, fontpath) == FALSE)
      PDEBUG(("Using default Font path '%s'", fontpath));
      
    for (i=0; i<2; i++) for (j=0; j<32; j++) font_table[i][j] = NULL; /* clear font table */

    if ((arg_str = strtok(findlabel(param_file, "FontSelect", LABEL_OPTIONAL+LABEL_FROMSTART)," ")) != NULL)
    {
      do
        {
          if ((tmpmem = (char *)malloc(256)) == NULL) PERROR(("Could not allocate 256 bytes!"));
          result=0;
          prev_arg_str = NULL;
          while ((arg_str=strtok(NULL," ")) != NULL)    /* get all font sizes from this one FontSelect line */
          {
            result++;
            if (prev_arg_str != NULL) ParseFontXY(prev_arg_str, &x, &y);
            font_table[x-8][y-1] = tmpmem;
            prev_arg_str = arg_str;
          }
          if (result <2) PERROR(("Incomplete 'FontSelect' line: needs at least one font size and a font name."));
          /* prev_arg_str should now be the font name */
          strcpy(tmpmem, prev_arg_str);
/*          PDEBUG(("FontSelect: Will use '%s' font file for all %dx%d modes",font_table[i][j], i+8, j+1)); */
        }
      while ((arg_str=strtok(findlabel(param_file, "FontSelect", LABEL_OPTIONAL+LABEL_FROMCURRENT)," ")) != NULL);
    }
  }

 /*
  * get cursor definition
  */
  change_cursor=FALSE;
  if (strtok(findlabel(param_file, "Cursor", LABEL_OPTIONAL+LABEL_FROMSTART)," ") != NULL)
  {
    change_cursor=TRUE;
    if ((arg_str=strtok(NULL," ")) == NULL)
      PERROR(("Cursor definition: no <start>-<end> position given in config file."))
    else
    {
      if (sscanf(arg_str, "%d-%d", &cursor_start, &cursor_end) != 2)
        PERROR(("Cursor definition: malformed <start>-<end>."));
      check_int_range(cursor_start, 0, 31, "cursor start line");
      check_int_range(cursor_end, cursor_start, 31, "cursor end line");
      /* scale the cursor to fit the chosen font size */
      cursor_start = (cursor_start * font_height) / 32;
      cursor_end = (cursor_end * font_height) / 32;
    }
  }
  else
  {
    /* default cursor size */
    cursor_start = font_height-2;
    cursor_end = font_height-1;
  }


 /*
  * look for a suitable clock for the specified chip set
  */
  if ((extclock == FALSE) && (clockchip < 0))
  { 
    GetClock(chipset, clocks, num_clocks, clockfreq, &realclock, optionmask);
    tempf = fabs(realclock-clockfreq);
    if( tempf > MAX_CLOCKDEVIATION )
      PERROR(("The closest available clock %f differs too much from specified clock %f",realclock,clockfreq));
    PDEBUG(("Clock deviation (from requested clock) will be |%5.2f-%5.2f|=%4.2f MHz (%4.2f%%).", clockfreq, realclock, tempf, (tempf*100)/clockfreq));
  }
  else realclock=clockfreq; /* suppose the external clock program or the clockchip code can do ANY clock */


 /*
  * calculate some screen parameters
  */
  textlines = activelines / font_height;
  activelines = font_height * textlines; /* make activelines integer multiple of font_height */
  blanklines = totallines - activelines;
  charsperline = (activepix / 8) & 0xFFFFFFFE; /* must be multiple of 2 in VGA byte-mode adressing */  

  horfreq = (realclock*1000)/( ((int)(totalpix / 8)) * font_width);
  vertfreq = (horfreq*1000)/totallines;
  
 /*
  * Now we should do some checking to see if the horizontal and vertical refresh frequencies are within limits
  */
  check_bounds_realtime(param_file, "HorizSync", horfreq, 30.0, 32.0, "Horizontal Sync Frequency");
  check_bounds_realtime(param_file, "VertRefresh", vertfreq, 50.0, 80.0, "Vertical Refresh Frequency");

 /*
  * start changing some things now.
  */

  if (program_hardware == TRUE)
  {
    /*
     * Get IO permissions first. No use in resizing when VGA programming is not allowed.
     */
     get_VGA_io_perm(chipset);

    /*
     * Resize the screen. Still needs LOTS more error checking to avoid dropping out in the middle, leaving
     * the user with a garbled screen.
     */

   #ifndef NO_VT_RESIZE
     /*
      * Resize the internal kernel VT parameters, so they comply with the new ones.
      * This is done BEFORE programming any hardware, so if it gives an "out of memory" error on heavily loaded machines,
      * or if it exits with an "ioctl: invalid argument" because the kernel doesn't support VT_RESIZE,
      * the screen isn't left all messed up. (suggested by Alessandro Rubini).
      *
      * First, get some memory, since the kernel cannot do a shrink_buffers() (=reduce file cache buffers in order to
      * get some more memory) while it is running with GFP_KERNEL priority to execute the VT_RESIZE. Resizing to large 
      * screens can require more memory than currently available, so if we malloc() some memory, and the free() it again, 
      * maybe it won't have been taken back by the time we do VT_RESIZE. (Thanks Mr Rubini ;-)
      *
      * Maybe we should disable interrupts between the free() and the VT_RESIZE, but it can't be done in user space.
      *
      */
      if (kversion >= 1001054)
      {
        PDEBUG(("Resizing kernel screen size parameters."));
        /* prepare parameters for VT_RESIZE */
        my_vt_size.v_rows = textlines;
        my_vt_size.v_cols = charsperline;
        my_vt_size.v_scrollsize = 0; /* kernel tries to get as many scroll-back lines as possible by itself (?) */

        /* get, and then free some memory, so VT_RESIZE doesn't fail */
        if ((tmpmem = (char *)malloc(512*1024)) == NULL)
          PERROR(("malloc(): Could not get temporary memory. Close some applications and try again..."));
        /* now write to the memory so it is "faulted in", otherwise it isn't really taken */
        for (i=0; i<512*1024; i+=1024) *(tmpmem+i) = (char)0;
        free(tmpmem);
        
        /* The Real Thing (TM) ... */
        /* first let the KERNEL know that the screen has been resized, so it writes the chars in the correct place */
        if (ioctl(0, VT_RESIZE, &my_vt_size))
        {
            perror("VT_RESIZE");
            PERROR(("Could not set kernel screen size parameters."));
        }
      }
      else PDEBUG(("Kernel version older than 1.1.54. VT_RESIZE not called."));

      /* now resize the TERMINALS (equivalent to "stty rows ... cols ...) , if any were given in a "Terminals" line 
         Another fine suggestion by Kenneth Albanowski */
      for (i=0; i<terminals; i++)
      {
        if ((dev = open(termstrings[i], O_RDWR | O_NOCTTY)) < 0)
        {
          perror("open");
          PERROR(("Could not open terminal device '%s' for writing.", termstrings[i]));
        }
        if (ioctl(dev, TIOCGWINSZ, &my_winsize))
        {
            perror("TIOCGWINSZ");
            PERROR(("Could not read Terminal size for %s", termstrings[i]));
        } 
        sresize=FALSE;
        if ((my_winsize.ws_col != charsperline) || (my_winsize.ws_row != textlines))
        {
          if (kversion < 1001054) PERROR(("Linux Kernel version %s does not support VGA screen resizing (must be >= 1.1.54).", my_utsname.release));
          sresize=TRUE; 
          PDEBUG(("TIOCGWINSZ: Resizing Terminal %s from %dx%d to %dx%d",
            termstrings[i], my_winsize.ws_col, my_winsize.ws_row, charsperline, textlines));
          my_winsize.ws_col = charsperline;
          my_winsize.ws_row = textlines;
          if (ioctl(dev, TIOCSWINSZ, &my_winsize))
          {
              perror("TIOCSWINSZ");
              PERROR(("Could not set Terminal size for %s", termstrings[i]));
          } 
        }
      }
    #else   /* no resizing support: do NOT allow resizing. */
        /* This is a bad hack: I use /dev/tty1 to get screen size. This MIGHT not be OK. */
        /* But you should have a newer kernel, so NO_VT_RESIZE is not needed ... */
        if ((dev = open("/dev/tty1", O_RDONLY | O_NOCTTY)) < 0)
        {
          perror("open");
          PERROR(("Could not open terminal device /dev/tty1 for reading."));
        }
        if (ioctl(dev, TIOCGWINSZ, &my_winsize))
        {
            perror("TIOCGWINSZ");
            PERROR(("Could not read Terminal size for /dev/tty1"));
        } 
        if ((my_winsize.ws_col != charsperline) || (my_winsize.ws_row != textlines))
          PERROR(("Resizing screens NOT allowed (NO_VT_RESIZE defined in source)"));
    #endif   

      
   /*
    * now get to the REAL hardware stuff !
    */
    vgaIOBase=GET_VGA_BASE;

    /*  WAIT_VERT_BLK; SCREEN_OFF; */

    unlock(chipset);

    if (run_pixclock==TRUE)
    {
      if (extclock == TRUE)
      {
         sprintf(tempstr,"%d %d", (int) (clockfreq*1000), extclockindex);
         SYNCRESET_SEQ;
         Run_extern_Prog("ClockProg", extclockpath, tempstr);
         ENDRESET_SEQ;
      }       
      else 
        if (clockchip >= 0) set_s3_clockchip_clock(clockchip, (long)(clockfreq*1000000), optionmask);
      else
      {
        /* if no clocks at this point, there must be a bug... (clockchip/clockprog already processed) */
        if (num_clocks <1) PERROR(("Internal error: No clock programming defined. Please complain to the author..."));
        SetClock(chipset, clocks, num_clocks, clockfreq, &realclock, optionmask);
      }
    }
    else PWARNING(("Clock will NOT be programmed! (due to command line switch '-c') !"));
    
    special(chipset); /* change chipset-specific things, if needed */

    Set_MAX_SCANLINE (font_height);

    Set_VERT_TOTAL(totallines);
    Set_VDISPL_END(activelines);
    Set_VRETRACE_START(start_vsync); Set_VRETRACE_END(stop_vsync);
    Set_VBLANK_START(start_vsync); Set_VBLANK_END(stop_vsync);

    Set_HOR_TOTAL(totalpix/8);
    Set_HOR_DISPL_END(charsperline);
    Set_HSYNC_START(start_hsync/8); Set_HSYNC_END(stop_hsync/8);
    Set_HBLANK_START((start_hsync-1)/8); Set_HBLANK_END(stop_hsync/8);
    Set_LOG_SCREEN_WIDTH(charsperline);

    Set_CURSOR_START(cursor_start) ; Set_CURSOR_END(cursor_end);

    Set_HSYNC_POLARITY(h_polarity) ; Set_VSYNC_POLARITY(v_polarity);

    Set_Textmode;  /* just in case some jerk set us in graphics mode. Or if he's in X-windows: surprise surprise ! */

    if (set_charwidth(font_width)) PERROR(("Illegal character width: %d",font_width));
    
    /* SCREEN_ON;  */

  }
  
 /*
  * show the user what he has just done
  */
  PDEBUG(("Refresh will be %6.3fkHz/%6.3fHz",horfreq,(horfreq*1000)/totallines));
  printf("Chipset = '%s', Textmode clock = %3.2f MHz, %dx%d chars, CharCell = %dx%d. Refresh = %3.2fkHz/%3.1fHz.\n",
          str_chipsets[chipset],realclock,charsperline,textlines,font_width,font_height,horfreq,vertfreq);

 /*
  * call the external font loading program, if enabled by the 'option loadfont' line
  */

 if ((optionmask & OPT_LOADFONT) && (program_hardware == TRUE) && (run_fontprog==TRUE))
 {
   if (font_table[font_width-8][font_height-1] == NULL)
   {
     PWARNING(("Font loading enabled, but no font specified for %dx%d text mode.", font_width, font_height));
     PWARNING(("No font will be loaded."));
   }
   else
   {
     sprintf(tempstr,"%s/%s", fontpath, font_table[font_width-8][font_height-1]);
     Run_extern_Prog("LoadFont", fontprogpath, tempstr);
   }
 }
 

 /*
  * call the reset program (which could be used to notify applications about the screen changes)
  * But only when the screen has been resized.
  */

 if ((resetprog == TRUE) && (program_hardware == TRUE) && (run_resetprog==TRUE))
 {
    if (sresize == FALSE)
      PDEBUG(("Screen not resized. ResetProg not executed."))
    else
    {
      sprintf(tempstr,"%d %d", charsperline, textlines);
      Run_extern_Prog("ResetProg", resetprogpath, tempstr);
    }
 }       

  return(0);
}

