/***
 *** VGA chip programming functions for SVGATextMode
 *** (c) 1995 Koen Gadeyne (kmg@barco.be)
 ***
 ***/

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#ifndef DOS
#include <unistd.h>
#include <asm/io.h>
#else
#include <dos.h>
#include <conio.h>
#define outb(data,port) outp(port,data)
#define outw(data,port) outpw(port,data)
#define inb(port) inp(port)
#define inw(port) inpw(port)
#define ioperm(x,y,z) (0)
#define usleep(x) delay(x/1000) /* Note lack of resolution */
#endif
#include <math.h>
#include "vga_prg.h"
#include "messages.h"  

/*
 * Some functions to make life easier (?!)
 */

void outb_ATR_CTL (int index, int data)
{
  inb(ATR_CTL_INDEX_DATA_SWITCH);
  outb (index & 0x1f, ATR_CTL_INDEX);
  outb ( data, ATR_CTL_DATA_W);
  inb(ATR_CTL_INDEX_DATA_SWITCH);
  outb ( (index & 0x1f) | 0x20, ATR_CTL_INDEX);
  outb ( data, ATR_CTL_DATA_W);
}

int inb_ATR_CTL (int index)
{
  int res;

  inb(ATR_CTL_INDEX_DATA_SWITCH);
  outb (index & 0x1f, ATR_CTL_INDEX);
  res=inb(ATR_CTL_DATA_R);
  inb(ATR_CTL_INDEX_DATA_SWITCH);
  outb ( (index & 0x1f) | 0x20, ATR_CTL_INDEX);
  inb(ATR_CTL_DATA_R);
  return res;
}

void outb_VGA_mem(int register_set, int data)
{
   switch(register_set)
  {
    case REGSET_MISC  : outb(data, VGA_MISC_W); break;
    default: PERROR(("outb_VGA_indexed: unknown register set %d",register_set));
  }
}

int inb_VGA_mem(int register_set)
{
   switch(register_set)
  {
    case REGSET_MISC  : return(inb(VGA_MISC_R)); break;
    default: PERROR(("inb_VGA_indexed: unknown register set %d",register_set));
  }
}

void outb_VGA_indexed(int register_set, int reg_index, int data)
{
   switch(register_set)
  {
    case REGSET_CRTC  : Outb_CRTC(reg_index,data); break;
    case REGSET_SEQ   : Outb_SEQ(reg_index,data); break;
    case REGSET_ATRCTL: outb_ATR_CTL(reg_index,data); break;
    case REGSET_GRCTL : Outb_GR_CTL(reg_index,data); break;
    default: PERROR(("outb_VGA_indexed: unknown register set %d",register_set));
  }
}

int inb_VGA_indexed(int register_set, int reg_index)
{
   switch(register_set)
  {
    case REGSET_CRTC  : return(Inb_CRTC(reg_index)); break;
    case REGSET_SEQ   : return(Inb_SEQ(reg_index)); break;
    case REGSET_ATRCTL: return(inb_ATR_CTL(reg_index)); break;
    case REGSET_GRCTL : return(Inb_GR_CTL(reg_index)); break;
    default: PERROR(("inb_VGA_indexed: unknown register set %d",register_set));
  }
}

/*****************************************************************************************************************************/
void get_VGA_io_perm(int chipset)
{
  int err=0;
  PDEBUG(("Getting VGA IO permissions for chipset #%d", chipset));
  err = (ioperm(0x3b4, 0x3df - 0x3b4 + 1, 1));
  if (err != 0)
  {
  perror("VGA I/O Permissions");
  PERROR(("Cannot get VGA I/O permissions: must be superuser or setuid root!"));
  }
  switch(chipset)
  {
    case CS_ATI:
    case CS_ATIMACH32:  err = ioperm(ATI_EXTREG, 2, 1);
                        break;
  }
  if (err != 0)
  {
  perror("VGA I/O Permissions");
  PERROR(("Cannot get special I/O permissions for chipset %d",chipset));
  }
  Renounce_SUID; /* if we are Setuid Root: renounce to further superuser rights (safer) */
}

/*****************************************************************************************************************************/

void unlock(int chipset)
{
   /* unlock ALL locked registers for specified chipset. A bit rough, but simplest */
   PDEBUG(("Unlocking chipset %d",chipset));
   Outbit_CRTC (0x11, 7, 0); /* CRTC index 0x00..0x07 */
   switch(chipset)
   {
    case CS_CIRRUS :
       Outb_SEQ (0x6, 0x12);	/* unlock cirrus special */
       break;
    case CS_S3     : 
       Outb_CRTC(0x39, 0xa5); /* system extension regs (CRTC index 0x50..0x5E) */
       Outb_CRTC(0x38, 0x48); /* S3 register set (CRTC index 0x30..0x3C) */
       Outbit_CRTC(0x35, 4, 0); /* VERT timing regs (CRTC index 6,7(bit0,2,3,5,7),9,10,11(bits0..3),15,16 ) */
       Outbit_CRTC(0x35, 5, 0); /* HOR timing regs (CRTC index 0..5, 17(bit2) ) */
      /*  Outbit_CRTC(0x40, 0, 1); */ /* enhanced register access, only for access to accelerator commands. Does NOT seem to work on my 805 */
       break;
    case CS_ET4000 : outb(0x03, 0x3BF); outb(0xA0, 0x3D8); /* ET4000W32i key */
                     break;
    case CS_ATI:
    case CS_ATIMACH32: ATI_PUTEXTREG(0xB4, ATI_GETEXTREG(0xB4) & 0x03);
                       ATI_PUTEXTREG(0xB8, ATI_GETEXTREG(0xb8) & 0xC0);
                       ATI_PUTEXTREG(0xB9, ATI_GETEXTREG(0xb9) & 0x7F);
                       ATI_PUTEXTREG(0xBE, ATI_GETEXTREG(0xbe) | 0x01);
                       break;
    default: PDEBUG(("UNLOCK VGA: No special register unlocking needed for chipset #%d",chipset));
   }
}
/*****************************************************************************************************************************/

void special(int chipset)
/* chipset specific settings, like memory speed and the likes */
{
   PDEBUG(("Setting chipset-specific special registers"));
   switch(chipset)
   {
     case CS_ATI    : ATI_PUTEXTREG(0xB0, ATI_GETEXTREG(0xb0) & ~0x08);   /* (for 188xx chips) Enable 8 CRT accesses for each CPU access */
                      break;
    default: PDEBUG(("SPECIAL VGA chip settings: no special settings for chipset #%d",chipset));
   }
}
/*****************************************************************************************************************************/

int set_charwidth(int width)
{
   switch(width)
   {
      case 8: Outbit_SEQ(1, 0, 1);
              SET_PIXELPAN(0);
              break;
      case 9: Outbit_SEQ(1, 0, 0);
              SET_PIXELPAN(8);
              break;
      default: return(1);
   }
   return(0);
}

int get_charwidth()
{
  return((Inb_SEQ(1) & 0x01) ? 8 : 9);
}

int Get_VERT_TOTAL()
{
  return( (Inb_CRTC(0x6) + ((Inb_CRTC(0x7) & 0x01) ? 256 : 0) + ((Inb_CRTC(0x7) & 0x20) ? 512 : 0)) + 2);
}

int Get_HOR_TOTAL()
{
  return(Inb_CRTC(0)+5);
}

int Get_HOR_DISPL_END()
{
  return((int)Inb_CRTC(1)+1);
}

int Get_HSYNC_START()
{
  return((int)Inb_CRTC(4));
}

int Get_HSYNC_END()
{
  int start, end;
  start = Get_HSYNC_START();
  end = Inb_CRTC(5) & 0x1f;
  return( ((((start & 0x1f) > end) ? (start + 0x20) : start) & 0xffffe0) | end );
}
 
int Get_VERT_DISPL_END()
{
  return( ( (int)Inb_CRTC(0x12) + ((Inb_CRTC(0x7) & 0x02) ? 256 : 0) + ((Inb_CRTC(0x7) & 0x40) ? 512 : 0) ) +1);
}

int Get_VRETRACE_START()
{
  return( (int)Inb_CRTC(0x10) + ((Inb_CRTC(0x7) & 0x04) ? 256 : 0) + ((Inb_CRTC(0x7) & 0x80) ? 512 : 0) );
}

int Get_VRETRACE_END()
{
  int start, end;
  start = Get_VRETRACE_START();
  end = Inb_CRTC(0x11) & 0x0f;
  return( ((((start & 0x0f) > end) ? (start + 0x10) : start) & 0xfffff0) | end );
}

int Get_MAX_SCANLINE()
{
  return((Inb_CRTC(0x9) & 0x1f) +1);
}

int Get_HSYNC_POLARITY()
{
  return( (inb(VGA_MISC_R) & 0x40) ? (-1) : 1);
}

int Get_VSYNC_POLARITY()
{
  return( (inb(VGA_MISC_R) & 0x80) ? (-1) : 1);
}

int Get_TX_GR_Mode()
/* returns 0 for text mode, 1 for graphics mode */
{
  return(Inb_GR_CTL(6) & 0x01);
}
 
