#ident "@(#)sendfax.c	1.57 94/04/13 Copyright (c) Gert Doering"
;
/* sendfax.c
 *
 * Send a Fax using a class 2 faxmodem.
 * Calls routines in faxrec.c and faxlib.c
 *
 * The code is still quite rough, but it works.
 */

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#ifndef _NOSTDLIB_H
#include <stdlib.h>
#endif
#ifndef sun
#include <sys/ioctl.h>
#endif
#include <signal.h>

#include "mgetty.h"
#include "tio.h"
#include "policy.h"
#include "fax_lib.h"

/* I don't know *why*, but the ZyXEL wants all bytes reversed */
#define REVERSE 1

char * fac_tel_no;
boolean	verbose = FALSE;

void exit_usage _P1( (program),
		     char * program )
{
    fprintf( stderr,
	     "usage: %s [options] <fax-number> <page(s) in g3-format>\n", program);
    fprintf( stderr,
	     "\tvalid options: -p, -h, -v, -l <device(s)>, -x <debug>, -n, -S\n");
    exit(1);
}

TIO fax_tio;
char * Device;

int fax_open_device _P2( (fax_tty, use_stdin),
			 char * fax_tty, boolean use_stdin )
{
    char	device[MAXPATH];
    int	fd;

    if ( use_stdin )			/* fax modem on stdin */
    {
	fd = 0;
	Device = ttyname(fd);		/* for faxrec() */
	if ( Device == NULL || *Device == '\0' ) Device = "unknown";
    }
    else
    {
	if ( verbose ) printf( "Trying fax device '/dev/%s'... ", fax_tty );

	if ( makelock( fax_tty ) != SUCCESS )
	{
	    if ( verbose ) printf( "locked!\n" );
	    lprintf( L_MESG, "cannot lock %s", fax_tty );
	    return -1;
	}
	
	sprintf( device, "/dev/%s", fax_tty );

	if ( ( fd = open( device, O_RDWR | O_NDELAY ) ) == -1 )
	{
	    lprintf( L_ERROR, "error opening %s", device );
	    if ( verbose ) printf( "cannot open!\n" );
	    rmlocks();
	    return fd;
	}

	/* make device name externally visible (faxrec()) */
	Device = fax_tty;
    }

    /* unset O_NDELAY (otherwise waiting for characters */
    /* would be "busy waiting", eating up all cpu) */

    if ( fcntl( fd, F_SETFL, O_RDWR ) == -1 )
    {
	lprintf( L_ERROR, "error in fcntl" );
	close( fd );
	if ( verbose ) printf( "cannot fcntl!\n" );
	rmlocks();
	return -1;
    }

    /* initialize baud rate, hardware handshake, ... */
    tio_get( fd, &fax_tio );

    /* even if we use a modem that requires Xon/Xoff flow control,
     * do *not* enable it here - it would interefere with the Xon
     * received at the top of a page.
     */
    tio_mode_sane( &fax_tio, TRUE );
    tio_set_speed( &fax_tio, FAX_SEND_BAUD );
    tio_mode_raw( &fax_tio );
#ifdef sun
    /* sunos does not rx with RTSCTS unless carrier present */
    tio_set_flow_control( fd, &fax_tio, FLOW_NONE );
#else
    tio_set_flow_control( fd, &fax_tio, (FAXSEND_FLOW) & FLOW_HARD );
#endif
    
    if ( tio_set( fd, &fax_tio ) == ERROR )
    {
	lprintf( L_ERROR, "error in tio_set" );
	close( fd );
	if ( verbose ) printf( "cannot set termio values!\n" );
	rmlocks();
	return -1;
    }

    /* reset parameters */
    fax_to_poll = FALSE;

    fax_remote_id[0] = 0;
    fax_param[0] = 0;

    if ( use_stdin )
    {
	lprintf( L_NOISE, "fax_open_device, fax on stdin" );
    }
    else
    {
	log_init_paths( NULL, NULL, &fax_tty[ strlen(fax_tty)-2 ] );
	lprintf( L_NOISE, "fax_open_device succeeded, %s -> %d", fax_tty, fd );
    }
    
    if ( verbose ) printf( "OK.\n" );
    return fd;
}

/* fax_open: loop through all devices in fax_ttys until fax_open_device()
 * succeeds on one of them; then return file descriptor
 * return "-1" of no open succeeded (all locked, permission denied, ...)
 */

int fax_open _P2( (fax_ttys, use_stdin),
	      char * fax_ttys, boolean use_stdin )
{
char * p, * fax_tty;
int fd;

    p = fax_tty = fax_ttys;
    do
    {
	p = strchr( fax_tty, ':' );
	if ( p != NULL ) *p = 0;
	fd = fax_open_device( fax_tty, use_stdin );
	if ( p != NULL ) *p = ':';
	fax_tty = p+1;
    }
    while ( p != NULL && fd == -1 );
    return fd;
}

/* finish off - close modem device, rm lockfile */

void fax_close _P1( (fd),
		    int fd )
{
    fax_send( "AT+FCLASS=0\r\n", fd );
    delay(100);
    close( fd );
    rmlocks();
}

RETSIGTYPE fax_sig_goodbye _P1( (signo), int signo )
{
    lprintf( L_AUDIT, "failed, pid=%d, got signal %d, exiting...", 
	     getpid(), signo );
    rmlocks();
    exit(15);				/* will close the fax device */
}

int main _P2( (argc, argv),
	      int argc, char ** argv )
{
int argidx;
int fd;
char buf[1000];
int	opt;
int i;

/* variables settable by command line options */
char *	extra_modem_init = NULL;
boolean fax_poll_wanted = FALSE;
char * 	fax_page_header = NULL;
char *	poll_directory = ".";			/* override with "-d" */

static char 	fax_device_string[] = FAX_MODEM_TTYS;	/* writable! */
char *	fax_devices = fax_device_string;	/* override with "-l" */
int	fax_res_fine = 1;			/* override with "-n" */

boolean	use_stdin = FALSE;			/* modem on stdin */

int	tries;

    /* initialize logging */
    log_init_paths( argv[0], FAX_LOG, NULL );
    log_set_llevel( L_NOISE );

    while ((opt = getopt(argc, argv, "d:vx:ph:l:nm:S")) != EOF) {
	switch (opt) {
	case 'd':	/* set target directory for polling */
	    poll_directory = optarg;
	    break;
	case 'v':	/* switch on verbose mode */
	    verbose = TRUE;
	    break;
	case 'x':	/* set debug level */
	    log_set_llevel( atoi(optarg) );
	    break;
	case 'p':	/* enable polling */
	    fax_poll_wanted = TRUE;
	    break;
	case 'h':	/* set page header */
	    fax_page_header = optarg;
	    lprintf( L_MESG, "page header: %s", fax_page_header );
	    break;
	case 'l':	/* set device(s) to use */
	    fax_devices = optarg;
	    if ( strchr( optarg, '/' ) != NULL )
	    {
		fprintf( stderr, "%s: -l: use device name without path\n",
		                 argv[0]);
		exit(1);
	    }
	    break;
	case 'n':	/* set normal resolution */
	    fax_res_fine = 0;
	    break;
	case 'm':
	    extra_modem_init = optarg;
	    break;
        case 'S':	/* modem on stdin */
	    use_stdin = TRUE;
	    break;
	case '?':	/* unrecognized parameter */
	    exit_usage(argv[0]);
	    break;
	}
    }

    argidx = optind;

    if ( argidx == argc )
    {
	exit_usage(argv[0]);
    }
    fac_tel_no = argv[ argidx++ ];

    lprintf( L_MESG, "sending fax to %s", fac_tel_no );

    if ( ! fax_poll_wanted && argidx == argc )
    {
	exit_usage(argv[0]);
    }

    /* check, if all the arguments passed are normal files and
     * readable
     */
    for ( i=argidx; i<argc; i++ )
    {
	lprintf( L_MESG, "checking %s", argv[i] );
	if ( access( argv[i], R_OK ) == -1 )
	{
	    lprintf( L_ERROR, "cannot access %s", argv[i] );
	    fprintf( stderr, "%s: cannot access %s\n", argv[0], argv[i]);
	    exit(1);
	}
    }

    if ( use_stdin ) verbose = FALSE;		/* no blurb to modem! */
    
    fd = fax_open( fax_devices, use_stdin );

    if ( fd == -1 )
    {
	lprintf( L_WARN, "cannot open fax device(s)" );
	fprintf( stderr, "%s: cannot access fax device(s) (locked?)\n", argv[0] );
	exit(2);
    }

    /* arrange that lock files get removed if INTR or QUIT is pressed */
    signal( SIGINT, fax_sig_goodbye );
    signal( SIGQUIT, fax_sig_goodbye );
    signal( SIGTERM, fax_sig_goodbye );

#ifdef HAVE_SIGINTERRUPT
    /* interruptible system calls */
    siginterrupt( SIGINT,  TRUE );
    siginterrupt( SIGALRM, TRUE );
    siginterrupt( SIGHUP,  TRUE );
#endif

    if ( fax_command( "AT", "OK", fd ) == ERROR ||
         fax_command( "AT+FCLASS=2", "OK", fd ) == ERROR )
    {
	lprintf( L_ERROR, "cannot set class 2 fax mode" );
	fprintf( stderr, "%s: cannot set class 2 fax mode\n", argv[0] );
	fax_close( fd );
	exit( 3 );
    }

#ifdef FAX_SEND_SWITCHBD
    /* some modems need a baud rate switch after +FCLASS=2,
     * see policy.h for details
     */
    tio_set_speed( &fax_tio, FAX_SEND_SWITCHBD );
    tio_set( fd, &fax_tio );
#endif
	
    sprintf( buf, "AT+FLID=\"%s\"", FAX_STATION_ID);
    if ( fax_command( buf, "OK", fd ) == ERROR )
    {
	lprintf( L_ERROR, "cannot set fax station ID" );
	fprintf( stderr, "%s: cannot set fax station ID\n", argv[0] );
	fax_close( fd );
	exit(3);
    }

    if ( extra_modem_init != NULL )
    {
	if ( strncmp( extra_modem_init, "AT", 2 ) != 0 )
	{
	    fax_send( "AT", fd );
	}

	if ( fax_command( extra_modem_init, "OK", fd ) == ERROR )
	{
	    lprintf( L_WARN, "cannot send extra modem init string '%s'",
		    extra_modem_init );
	    fprintf( stderr, "%s: modem doesnt't accept '%s'\n",
		    argv[0], extra_modem_init );
	    fax_close( fd );
	    exit(3);
	}
    }		/* end if (extra_modem_init != NULL) */

    /* FIXME: ask modem if it can do 14400 bps / fine res. at all */

    sprintf( buf, "AT+FDCC=%d,5,0,2,0,0,0,0", fax_res_fine );
    fax_command( buf, "OK", fd );

#if REVERSE
    fax_command( "AT+FBOR=0", "OK", fd );
#else
    fax_command( "AT+FBOR=1", "OK", fd );
#endif

    /* tell the modem if we are willing to poll faxes
     */
    if ( fax_poll_wanted )
    {
	sprintf( buf, "AT+FCIG=\"%s\"", FAX_STATION_ID);
	if ( fax_command( buf, "OK", fd ) == ERROR ||
	     fax_command( "AT+FSPL=1", "OK", fd ) == ERROR )
	{
	    lprintf( L_WARN, "AT+FSPL=1: cannot enable polling" );
	    fprintf( stderr, "Warning: polling is not possible!\n" );
	    fax_poll_wanted = FALSE;
	    fax_hangup = 0;	/* reset error flag */
	}
    }

    /* set modem to use desired flow control type, dial out
     */
    if ( verbose ) { printf( "Dialing %s... ", fac_tel_no ); fflush(stdout); }

    sprintf( buf, "AT%sD%s", FAX_MODEM_HANDSHAKE, fac_tel_no );
    if ( fax_command( buf, "OK", fd ) == ERROR )
    {
	lprintf( L_WARN, "dial failed (hangup_code=%d)", fax_hangup_code );
	fprintf( stderr, "\n%s: dial %s failed (%s)\n", argv[0], fac_tel_no,
		 fax_hangup_code == FHUP_BUSY? "BUSY" : "ERROR / NO CARRIER");

	/* end program - return codes signals kind of dial failure */
	fax_close( fd );

	if ( fax_hangup_code == FHUP_BUSY ) exit(4);
	                               else exit(10); 
    }
    if ( verbose ) printf( "OK.\n" );

#ifndef FAX_SEND_IGNORE_CARRIER
    /* by now, the modem should have raised DCD, so remove CLOCAL flag */
    tio_carrier( &fax_tio, TRUE );

#ifdef sun
    /* now we can request hardware flow control since we have carrier */
    tio_set_flow_control( fd, &fax_tio, (FAXSEND_FLOW) & (FLOW_HARD|FLOW_XON_OUT) );
#endif	/* sun */
    tio_set( fd, &fax_tio );

#endif	/* !FAX_SEND_IGNORE_CARRIER */

    /* process all files to send / abort, if Modem sent +FHNG result */

    tries = 0;
    while ( argidx < argc )
    {
	/* send page header, if requested */
	if ( fax_page_header )
	{
#if 0
	    if ( fax_send_page( fax_page_header, fd ) == ERROR ) break;
#else
	    fprintf( stderr, "WARNING: no page header is transmitted. Does not work yet!\n" );
#endif
	    /* NO page punctuation, we want the next file on the same page
	     */
	}

	/* send page */
	if ( verbose ) printf( "sending '%s'...\n", argv[ argidx ] );
	if ( fax_send_page( argv[ argidx ], &fax_tio, fd ) == ERROR ) break;

	fax_page_tx_status = -1;

        /* transmit page punctuation
	 * (three cases: more pages, last page but polling, last page at all)
	 * then evaluate +FPTS: result code
	 */

	if ( argidx == argc-1 )		/* was this the last page to send? */
	  if ( fax_poll_wanted && fax_to_poll )
	    fax_command( "AT+FET=1", "OK", fd );	/* end document */
	  else
	  {
	    /* take care of some modems pulling cd low too soon */
	    tio_carrier( &fax_tio, FALSE );
#ifdef sun
	    /* HW handshake has to be off while carrier is low */
	    tio_set_flow_control(fd, &fax_tio, (FAXSEND_FLOW) & FLOW_XON_OUT);
#endif
	    tio_set( fd, &fax_tio );

	    fax_command( "AT+FET=2", "OK", fd );	/* end session */
	  }
	else
	    fax_command( "AT+FET=0", "OK", fd );	/* end page */

	/* after the page punctuation command, the modem
	 * will send us a +FPTS:<ppm> page transmit status.
	 * The ppm value is written to fax_page_tx_status by
	 * fax_wait_for()
	 * If the other side requests retransmission, do so.
	 */

	switch ( fax_page_tx_status )
	{
	    case 1: tries = 0; break;		/* page good */
						/* page bad - r. req. */
	    case 2:
#if FAX_SEND_MAX_TRIES <= 0
	      fprintf( stderr, "WARNING: page bad (+FTPS:2), ignoring\n" );
	      lprintf( L_WARN, "WARNING: +FPTS:2 ignored\n" );
#else	      
	      fprintf( stderr, "ERROR: page bad - retrain requested\n" );
	      tries ++;	
	      if ( tries >= FAX_SEND_MAX_TRIES )
	      {
		  fprintf( stderr, "ERROR: too many retries - aborting send\n" );
		  fax_hangup_code = -1;
		  fax_hangup = 1;
	      }
	      else
	      {
		  if ( verbose )
		  printf( "sending page again (retry %d)\n", tries );
		  continue;	/* don't go to next page */
	      }
#endif	/* FAX_SEND_MAX_TRIES > 0 */
	      break;
	    case 3: fprintf( stderr, "WARNING: page good, but retrain requested\n" );
		    break;
	    case 4:
	    case 5: fprintf( stderr, "WARNING: procedure interrupt requested - don't know how to handle it\n" );
		    break;
	    case -1:			/* something broke */
		    lprintf( L_WARN, "fpts:-1" );
		    break;
	    default:fprintf( stderr, "WARNING: invalid code: +FPTS:%d\n",
	                             fax_page_tx_status );
		    break;
	}

	if ( fax_hangup && fax_hangup_code != 0 ) break;
	
	argidx++;
    }				/* end main page loop */

    if ( argidx < argc || ( fax_hangup && fax_hangup_code != 0 ) )
    {
	lprintf( L_WARN, "Failure transmitting %s: +FHNG:%2d",
		 argv[argidx], fax_hangup_code );
	fprintf( stderr, "\n%s: FAILED to transmit '%s'.\n",
		         argv[0], argv[argidx] );

	if ( fax_hangup_code == -1 )
	    fprintf( stderr, "(number of tries exhausted)\n" );
	else
	    fprintf( stderr, "Transmission error: +FHNG:%2d (%s)\n",
			     fax_hangup_code,
			     fax_strerror( fax_hangup_code ) );
	fax_close( fd );
	exit(12);
    }

    /* OK, handle (optional) fax polling now.
     * Fax polling will only be tried if user specified "-p" and the
     * faxmodem sent us a "+FPOLL" response
     */

    if ( fax_poll_wanted )
    {
    int pagenum = 0;

	if ( verbose ) printf( "starting fax poll\n" );

	if ( ! fax_to_poll )
	{
	    printf( "remote does not have document to poll!\n" );
	}
	else
	{
	    /* switch to fax receiver flow control */
	    tio_set_flow_control( fd, &fax_tio,
				 (FAXREC_FLOW) & (FLOW_HARD|FLOW_XON_IN) );
	    tio_set( fd, &fax_tio );
	    if ( fax_get_pages( fd, &pagenum, poll_directory ) == ERROR )
	    {
		fprintf( stderr, "warning: polling failed\n" );
		lprintf( L_WARN, "warning: polling failed!" );
		fax_close( fd );
		exit(12);
	    }
	}
	if ( verbose ) printf( "%d pages successfully polled!\n", pagenum );
    }

    fax_close( fd );
    return 0;
}
