/* Copyright (C) 1991, 1992, 1995 Free Software Foundation, Inc.
This file is part of the GNU C Library.

The GNU C Library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

The GNU C Library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with the GNU C Library; see the file COPYING.LIB.  If
not, write to the Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA.  */

#include <alloca.h>
#include <stdio.h>
#include <printf.h>
#include <stdlib.h>
#include <string.h>

#include "printf-parse.h"


size_t
parse_printf_format (fmt, n, argtypes)
      const char *fmt;
      size_t n;
      int *argtypes;
{
  /* Count number of specifications.  */
  size_t nspecs;
  size_t nspecs_max;
  struct printf_spec *specs;
  /* Number of arguments.  */
  size_t nargs;
  /* Number of maximal referenced argument in positional parameter.  */
  size_t max_ref_arg;
  /* Just a counter.  */
  size_t cnt;

  /* Search for first format specification.  */
  fmt = find_spec (fmt);

  specs = (struct printf_spec *) alloca ((nspecs_max = 32)
					 * sizeof (struct printf_spec));
  memset (specs, 0, nspecs_max * sizeof (struct printf_spec));
  nspecs = 0;
  nargs = 0;
  max_ref_arg = 0;

  while (*fmt != '\0')
    {
      if (nspecs >= nspecs_max)
        /* Extend the arrays for the format specifiers.  */
        {
          struct printf_spec *old = specs;

          specs = (struct printf_spec *) alloca ((nspecs_max <<= 1)
                                                * sizeof (struct printf_spec));
	  memset (specs, 0, nspecs_max * sizeof (struct printf_spec));
          memcpy (specs, old, nspecs * sizeof (struct printf_spec));
        }

      nargs += parse_one_spec (fmt, nargs, specs + nspecs, &max_ref_arg);

      fmt = specs[nspecs++].next_fmt;
    }

  /* Now determine the types for all arguments requested.  */
  for (cnt = 0; cnt < nspecs; ++cnt)
    {
      /* If the width is determined by an argument this is an int.  */
      if (specs[cnt].width_arg != -1 && (size_t) specs[cnt].width_arg < n)
	argtypes[specs[cnt].width_arg] = PA_INT;

      /* If the precision is determined by an argument this is an int.  */
      if (specs[cnt].prec_arg != -1 && (size_t) specs[cnt].prec_arg < n)
	argtypes[specs[cnt].prec_arg] = PA_INT;

      if (specs[cnt].ndata_args == 0)
	/* Some formats don't use any parameter at all.  */
	continue;
      else if (specs[cnt].ndata_args == 1)
	/* We have only one argument.  The type is already determined.  */
	{
	  if ((size_t) specs[cnt].data_arg < n)
	    argtypes[specs[cnt].data_arg] = specs[cnt].data_arg_type;
	}
      else
	/* We have more than one argument for this format specification.  */
	{
	  /* This is defined in reg-printf.c.  */
	  extern printf_arginfo_function __printf_arginfo_table [];

	  /* Perhaps the arguments for this format are out of the range
	     in which we are interested.  */
	  int to_get = MIN (n - specs[cnt].data_arg - 1,
			    (size_t) specs[cnt].ndata_args);

	  if (to_get > 0)
	    (void) (*__printf_arginfo_table[specs[cnt].info.spec])
		(&specs[cnt].info, to_get, &argtypes[specs[cnt].data_arg]);
	}

    }

  return MAX (nargs, max_ref_arg);
}
