/*
 * ifconfig	This file contains an implementation of the command
 *		that either displays or sets the characteristics of
 *		one or more of the system's networking interfaces.
 *
 * Usage:	ifconfig [-v] interface [address] [up | down]
 * 		ifconfig [-v] interface net [addr]
 * 		ifconfig [-v] interface route [addr]
 *
 * Version:	@(#)ifconfig.c	1.04	03/03/93
 *
 * Author:	Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/ddi.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <netinet/in.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <resolv.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>


#define _PATH_IFACES	"/proc/net/dev"


static char *Version = "@(#) ifconfig 1.03 (03/03/93)";


int opt_v = 0;				/* debugging output flag	*/


static struct in_addr get_addr(char *name);
static struct in_addr get_net(char *name);


static struct in_addr get_addr(name)
char *name;
{
  struct hostent *hp;
  struct in_addr in;

  if ((hp = gethostbyname(name)) == (struct hostent *)NULL) {
	in.s_addr = INADDR_ANY;
	return(in);
  }
  memcpy((char *) &in, (char *) hp->h_addr_list[0], hp->h_length);
  strcpy(name, hp->h_name);
  return(in);
}


static struct in_addr get_net(name)
char *name;
{
  struct netent *np;
  struct in_addr in;

  /* First off: try a host name. */
  in = get_addr(name);
  if (in.s_addr != INADDR_ANY) return(in);

  /* No go.  Try a network name... */
  if ((np = getnetbyname(name)) == (struct netent *)NULL) {
	in.s_addr = INADDR_ANY;
	return(in);
  }
  in.s_addr = htonl(np->n_net);
  strcpy(name, np->n_name);
  return(in);
}


static void pr_ifaces(void)
{
  char buff[1024];
  int fd, len;

  if ((fd = open(_PATH_IFACES, O_RDONLY)) < 0) {
	perror(_PATH_IFACES);
	return;
  }

  while((len = read(fd, buff, 1024)) > 0)
		(void) write(1, buff, len);

  (void) close(fd);
}


static void usage()
{
  fprintf(stderr, "Usage: ifconfig interface [address] [up] [down]\n");
  fprintf(stderr, "       ifconfig interface net <addr>\n");
  fprintf(stderr, "       ifconfig interface route <addr>\n");
  exit(-1);
}


void main(argc, argv)
int argc;
char *argv[];
{
  char hostname[128];
  struct in_addr in;
  struct ip_config ifa;
  register int c, s;
  extern int getopt(), optind, opterr;

  /* Fetch the command-line arguments. */
  opterr = 0;
  while ((c = getopt(argc, argv, "v")) != EOF) switch(c) {
	case 'v':
		opt_v = 1;
		break;
	default:
		usage();
  }

  /* Fetch the interface name. */
  if (optind >= argc) {
	pr_ifaces();
	exit(0);
  }

  strncpy(ifa.name, argv[optind++], MAX_IP_NAME);
  ifa.up = 0;
  ifa.paddr = 0L;
  ifa.net = -1L;
  ifa.router = -1L;
  if (opt_v) fprintf(stderr, "IFA.NAME=\"%s\"\n", ifa.name);
 
  /* Create a socket to the INET kernel. */
  if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	perror("socket");
	exit(-1);
  }

  /* Fetch the interface's hostname or IP address. */
  strcpy(hostname, argv[optind]);
  in = get_addr(hostname);
  if (in.s_addr == INADDR_ANY) {
	fprintf(stderr, "ifconfig: unknown host %s\n", argv[optind]);
	exit(-1);
  }
  ifa.paddr = in.s_addr;
  if (opt_v) fprintf(stderr, "\"%s\": \"%s\" (%s)\n",
			argv[optind], hostname, inet_ntoa(in));

  /* Scan the argument list. */
  while(argv[++optind] != (char *)NULL) {
	if (!strcmp(argv[optind], "up")) {
		if (opt_v) fprintf(stderr, ">> UP\n");
		ifa.up = 1;
	} else if (!strcmp(argv[optind], "down")) {
		if (opt_v) fprintf(stderr, ">> DOWN\n");
		ifa.up = 0;
	} else if (!strcmp(argv[optind], "net")) {
		if (opt_v) fprintf(stderr, ">> NET: ");
		strcpy(hostname, argv[++optind]);
		in = get_net(hostname);
		if (in.s_addr == INADDR_ANY) {
			if (opt_v) fprintf(stderr,
				"NET FAILED\n          HOST: ");
			strcpy(hostname, argv[optind]);
			in = get_addr(hostname);
		}
		if (in.s_addr == INADDR_ANY) {
			fprintf(stderr, "ifconfig: unknown network %s\n",
								argv[optind]);
			exit(-1);
		}
		ifa.net = in.s_addr;
		if (opt_v) fprintf(stderr, "%s --> %s\n",
					hostname, inet_ntoa(in));
	} else if (!strcmp(argv[optind], "route")) {
		if (opt_v) fprintf(stderr, ">> ROUTE: ");
		strcpy(hostname, argv[++optind]);
		in = get_addr(hostname);
		if (in.s_addr == INADDR_ANY) {
			fprintf(stderr, "ifconfig: unknown host %s\n",
								argv[optind]);
			exit(-1);
		}
		ifa.router = in.s_addr;
		if (opt_v) fprintf(stderr, "%s --> %s\n",
					hostname, inet_ntoa(in));
	} else usage();
  }

  if (ioctl(s, IP_SET_DEV, &ifa) < 0) {
	perror("ioctl SETDEV");
	(void) close(s);
	exit(-1);
  }

  /* Close the socket. */
  (void) close(s);

  exit(0);
}
