/*
 * ifsetup	This file contains an implementation of the command
 *		that configures a kernel-resident DDI device driver
 *		to a working state.  This includes setting the I/O
 *		address, the IRQ and that sort of thing.
 *
 * Usage:	ifsetup [-v] filename [command=value] ...
 *
 * Version:	@(#)ifsetup.c	1.00	02/14/93
 *
 * Author: 	Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/ddi.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>


static char *Version = "@(#) ifsetup 1.00 (02/14/93)";


int opt_v = 0;				/* debugging output flag	*/


/* Convert an ASCII number to decimal. */
long atohex(char *sp)
{
  unsigned long val;
  int base = 10;

  /* Is it either one of the "special" values? */
  if (!strcmp(sp, "clear")) return(0L);
  if (!strcmp(sp, "auto")) return(-1L);

  /* No.  It must be a normal value then. */
  val = 0;
  while(*sp) {
	if (*sp == '0') {
		if (*(sp + 1) == 'x' || *(sp + 1) == 'X') {
			base = 16;
			sp += 2;
		} else {
			base = 8;
			sp++;
		}
		continue;
	}
	val *= base;
	switch(base) {
		case 8:
			if (*sp >= '0' && *sp <= '7') val += (*sp - '0');
			  else return(0L);
			break;
		case 10:
			if (*sp >= '0' && *sp <= '9') val += (*sp - '0');
			  else return(0L);
			break;
		case 16:
			if (*sp >= '0' && *sp <= '9') val += (*sp - '0');
			  else if (*sp >= 'a' && *sp <= 'f') val += (*sp - 'a');
			  else if (*sp >= 'A' && *sp <= 'F') val += (*sp - 'F');
			  else return(0L);
			break;
	}
	sp++;
  }
  return(val);
}

		
/* Show the current I/O configuration of a DDI device. */
void show_dev(char *name, int dev)
{
  struct ddconf conf;

  printf("\t**** %s ****\n\n", name);

  if (ioctl(dev, DDIOCGCONF, &conf) < 0) {
	perror("DDIOCGCONF");
	return;
  }

  printf("I/O address: 0x%X\n", conf.ioaddr);
  printf("AUX address: 0x%X\n", conf.ioaux);
  printf("IRQ channel: %d\n", (int)conf.irq);
  printf("DMA channel: %d\n", (int)conf.dma);
  printf("RAM address: 0x%X\n", conf.memaddr);
  printf("RAM size   : 0x%X (%d)\n", conf.memsize, conf.memsize);
}


/* Set the new I/O configuration of a DDI device. */
void conf_dev(char *name, int dev, char **args)
{
  struct ddconf conf;
  register char *sp;

  /* Fetch the current setup. */
  if (ioctl(dev, DDIOCGCONF, &conf) < 0) {
	perror("DDIOCGCONF");
	return;
  }

  while (*args != NULL) {
	if ((sp = strchr(*args, '=')) == NULL) {
		fprintf(stderr, "ifsetup: syntax error: \"%s\"\n", *args);
		args++;
		continue;
	}
	*sp++ = '\0';
	if (!strcmp(*args, "io")) conf.ioaddr = (int) atohex(sp);
	  else if (!strcmp(*args, "aux")) conf.ioaux = (int) atohex(sp);
	  else if (!strcmp(*args, "irq")) conf.irq = (short int) atohex(sp);
	  else if (!strcmp(*args, "dma")) conf.dma = (short int) atohex(sp);
	  else if (!strcmp(*args, "maddr")) conf.ioaddr = atohex(sp);
	  else if (!strcmp(*args, "msize")) conf.ioaddr = atohex(sp);
	  else {
		fprintf(stderr, "ifsetup: unknown parameter: \"%s\"\n", *args);
		args++;
		continue;
	}
	args++;
  }

  /* Save the new setup. */
  if (ioctl(dev, DDIOCSCONF, &conf) < 0) {
	perror("DDIOCSCONF");
	return;
  }
}


void usage(void)
{
  fprintf(stderr, "Usage: ifsetup [-v] filename [command=value] ...\n");
  exit(-1);
}


void main(argc, argv)
int argc;
char *argv[];
{
  char if_name[DDI_MAXNAME + 1];
  register int c;
  int dev;
  char *dev_name;
  extern int getopt(), optind, opterr;

  /* Initialize variables... */
  dev = -1;

  /* Fetch the command-line arguments. */
  opterr = 0;
  while ((c = getopt(argc, argv, "v")) != EOF) switch(c) {
	case 'v':
		opt_v = 1;
		break;
	default:
		usage();
  }

  /* At least one more argument needed! */
  if (optind == argc) usage();
  dev_name = argv[optind++];

  /* Open the device file. */
  if ((dev = open(dev_name, O_RDWR)) < 0) {
	perror(dev_name);
	exit(-1);
  }

  /* Ask the device for its DDI ID string. */
  if (ioctl(dev, DDIOCGNAME, if_name) < 0) {
	perror("DDIOCGNAME");
	(void) close(dev);
	exit(-1);
  }

  if (optind != argc) conf_dev(if_name, dev, &argv[optind]);
    else show_dev(if_name, dev);

  /* All done.  Close up and exit. */
  (void) close(dev);
  exit(0);
}
