/*
 * route	This file contains an implementation of the command
 *		that manages the IP routing table in the kernel.
 *
 * Usage:	route [-v] [ {add|del} target iface [ gw ] [ metric ] ]
 *
 * Version:	@(#)route.c	1.02	04/17/93
 *
 * Author:	Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <netinet/in.h>
#include <net/if.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <resolv.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>


#define _PATH_ROUTES	"/proc/net/route"


static char *Version = "@(#) route 1.02 (04/17/93)";


int opt_v = 0;				/* debugging output flag	*/
int sock = -1;


int resolve(char *name, struct in_addr *in)
{
  struct hostent *hp;
  struct netent *np;

  /* Default is special, meaning 0.0.0.0. */
  if (!strcmp(name, "default")) {
	memset((char *) in, 0, sizeof(struct in_addr));
	return(1);
  }

  /* Try the NETWORKS database to see if this is a known network. */
  if ((np = getnetbyname(name)) != (struct netent *)NULL) {
	in->s_addr = htonl(np->n_net);
	strcpy(name, np->n_name);
	return(1);
  }

  if (opt_v) {
	_res.options |= RES_DEBUG;
	res_init();
  }

  if ((hp = gethostbyname(name)) == (struct hostent *)NULL) {
	return(-1);
  }
  memcpy((char *) in, (char *) hp->h_addr_list[0], hp->h_length);
  strcpy(name, hp->h_name);
  return(0);
}


static void rt_print(void)
{
  char buff[1024];
  int fd, len;

  if ((fd = open(_PATH_ROUTES, O_RDONLY)) < 0) {
	perror(_PATH_ROUTES);
	return;
  }

  while((len = read(fd, buff, 1024)) > 0)
		(void) write(1, buff, len);

  (void) close(fd);
}


/* Add a routing table entry. */
int rt_add(char **args)
{
  char target[128], gateway[128];
  struct rtreq rt;
  struct sockaddr_in trg, gw;
  char *iface;
  int isnet;

  strcpy(target, *args++);
  iface = *args++;

  if ((isnet = resolve(target, &trg.sin_addr)) < 0) {
	herror(target);
	return(-1);
  }
  trg.sin_family = AF_INET;

  /* Clean out the RTREQ structure. */
  memset((char *) &rt, 0, sizeof(struct rtreq));
  rt.r_flags = (RTF_UP | RTF_HOST);
  if (isnet) rt.r_flags &= ~RTF_HOST;
  strncpy(rt.r_dev, iface, IFNAMSIZ);
  memcpy((char *) &rt.r_dst, (char *) &trg, sizeof(struct sockaddr));

  /* Did we specify a GATEWAY entry? */
  if ((*args != (char *)NULL) && (!strcmp(*args, "gw"))) {
	strcpy(gateway, *++args);
	if ((isnet = resolve(gateway, &gw.sin_addr)) < 0) {
		herror(gateway);
		return(-1);
	}
	if (isnet) {
		fprintf(stderr, "%s: cannot use a NETWORK as gateway!\n",
								gateway);
		return(-1);
	}
	gw.sin_family = AF_INET;
	memcpy((char *) &rt.r_gateway, (char *) &gw, sizeof(struct sockaddr));
	rt.r_flags |= RTF_GATEWAY;
	args++;
  } else strcpy(gateway, "NONE");

  /* Did we specify a METRIC field? */
  if ((*args != (char *)NULL) && (!strcmp(*args, "metric"))) {
	rt.r_metric = atoi(*++args);
  }

  if (opt_v) {
	printf("ROUTE ADD:\n---\n");
	printf("\tTARGET  = %s (%s)\n", target,
		inet_ntoa((*(struct sockaddr_in *) &rt.r_dst).sin_addr));
	printf("\tGATEWAY = %s (%s)\n", gateway,
		inet_ntoa((*(struct sockaddr_in *) &rt.r_gateway).sin_addr));
	printf("\tDEVICE  = %s\n", rt.r_dev);
	printf("\tMETRIC  = %d\n", rt.r_metric);
	printf("\tFLAGS   = ");
	if (rt.r_flags & RTF_UP) printf("UP ");
	if (rt.r_flags & RTF_GATEWAY) printf("GW ");
	if (rt.r_flags & RTF_HOST) printf("HOST ");
	  else printf("NET ");
	printf("\n---\n");
  }

  /* Tell the kernel to accept this route. */
  if (ioctl(sock, SIOCADDRT, &rt) < 0) {
	fprintf(stderr, "SIOCADDRT: %s\n", strerror(errno));
	return(-1);
  }

  return(0);
}


/* Delete a routing table entry. */
int rt_del(char **args)
{
  char target[128];
  struct rtreq rt;
  struct sockaddr_in trg;

  strcpy(target, *args++);

  if (resolve(target, &trg.sin_addr) < 0) {
	herror(target);
	return(-1);
  }
  trg.sin_family = AF_INET;

  /* Clean out the RTREQ structure. */
  memset((char *) &rt, 0, sizeof(struct rtreq));
  memcpy((char *) &rt.r_dst, (char *) &trg, sizeof(struct sockaddr));

  if (opt_v) {
	printf("ROUTE DEL:\n---\n");
	printf("\tTARGET  = %s (%s)\n", target,
		inet_ntoa((*(struct sockaddr_in *) &rt.r_dst).sin_addr));
	printf("---\n");
  }

  /* Tell the kernel to delete this route. */
  if (ioctl(sock, SIOCDELRT, &rt) < 0) {
	fprintf(stderr, "SIOCDELRT: %s\n", strerror(errno));
	return(-1);
  }
  return(0);
}


static void usage(void)
{
  fprintf(stderr, "Usage: route [-v] [ {add|del} target iface ");
  fprintf(stderr, "[ gw ] [ metric ] ]\n");
  exit(-1);
}


void main(argc, argv)
int argc;
char *argv[];
{
  register int c;
  extern int getopt(), optind, opterr;

  /* Fetch the command-line arguments. */
  opterr = 0;
  while ((c = getopt(argc, argv, "v")) != EOF) switch(c) {
	case 'v':
		opt_v = 1;
		break;
	default:
		usage();
  }

  /* Do we have to show the contents of the routing table? */
  if (optind == argc) {
	rt_print();
	exit(0);
  }

  /* Fetch the command. */
  if ((optind >= argc) ||
      (strcmp(argv[optind], "add") && strcmp(argv[optind], "del")) ||
      (optind >= (argc - 1))) usage();

  /* Create a socket to the INET kernel. */
  if ((sock = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	perror("socket");
	exit(-1);
  }

  /* See what we have to do here. */
  if (!strcmp(argv[optind++], "add")) {
	if (optind >= (argc - 1)) usage();
	c = rt_add(&argv[optind]);
  } else c = rt_del(&argv[optind]);

  /* Close the socket. */
  (void) close(sock);

  exit(c);
}
