/*
 * wdsetup	This program can be used to (re-)configure an Ethernet
 *		card of the Western Digital WD80x3 and SMC Elite series.
 *
 * Usage:	wdsetup [-e] [-r] [-v] [io_addr]
 *
 * Version:	@(#)wdsetup.c	0.03	02/08/93
 *
 * Authors:	A George Kalwitz Production, 1990
 *		Gregg Weber, <gregg@netcom.com>
 *	 	Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>


/******************************************************************************
 BOARD ID DEFINITIONS

 32 Bits of information are returned by 'GetBoardID ()'.

	The low order 16 bits correspond to the Feature Bits which make
	up a unique ID for a given class of boards.

		e.g. STARLAN MEDIA, INTERFACE_CHIP

		note: board ID should be ANDed with the STATIC_ID_MASK
		      before comparing to a specific board ID


	The high order 16 bits correspond to the Extra Bits which do not
	change the boards ID.

		e.g. INTERFACE_584_CHIP, 16 BIT SLOT, ALTERNATE IRQ

******************************************************************************/
#define BID_EEPROM_OVERRIDE     0xffd0ffb0      /* stuff override by eeprom */
#define	STARLAN_MEDIA		0x00000001	/* StarLAN		*/
#define	ETHERNET_MEDIA		0x00000002	/* Ethernet		*/
#define	TWISTED_PAIR_MEDIA	0x00000003	/* Twisted Pair		*/
#define	EW_MEDIA		0x00000004	/* Ethernet and Twisted Pair */
#define	INTERFACE_CHIP		0x00000010	/* Soft Config Adapter	*/
#define	BID_UNUSED_1		0x00000020	/* used to be INTELLIGENT */
#define	BOARD_16BIT		0x00000040	/* 16 bit capability	*/
#define	RAM_SIZE_UNKNOWN	0x00000000	/* 000 => Unknown RAM Size */
#define	RAM_SIZE_RESERVED_1	0x00010000	/* 001 => Reserved	*/
#define	RAM_SIZE_8K		0x00020000	/* 010 => 8k RAM	*/
#define	RAM_SIZE_16K		0x00030000	/* 011 => 16k RAM	*/
#define	RAM_SIZE_32K		0x00040000	/* 100 => 32k RAM	*/
#define	RAM_SIZE_64K		0x00050000	/* 101 => 64k RAM	*/ 
#define	RAM_SIZE_RESERVED_6	0x00060000	/* 110 => Reserved	*/ 
#define	RAM_SIZE_RESERVED_7	0x00070000	/* 111 => Reserved	*/ 
#define	BID_UNUSED_2		0x00010000	/* utb: RAM Size field	*/
#define	BID_UNUSED_3		0x00020000	/* utb: RAM Size field	*/
#define	BID_UNUSED_4		0x00040000	/* utb: RAM Size field	*/
#define	SLOT_16BIT		0x00080000	/* 16 bit board - 16 bit slot */
#define	NIC_690_BIT		0x00100000	/* NIC is 690		*/
#define	ALTERNATE_IRQ_BIT	0x00200000	/* Alternate IRQ is used */
#define	INTERFACE_5X3_CHIP	0x00000000	/* 0000 = 583 or 593 chips */
#define	INTERFACE_584_CHIP	0x00400000	/* 0001 = 584 chip	*/
#define	INTERFACE_594_CHIP	0x00800000	/* 0010 = 594 chip	*/

#define	MEDIA_MASK		0x00000007	/* Isolates Media Type	*/
#define	RAM_SIZE_MASK		0x00070000	/* Isolates RAM Size	*/
#define	STATIC_ID_MASK		0x0000FFFF	/* Isolates Board ID	*/
#define	INTERFACE_CHIP_MASK	0x03C00000	/* Isolates Intfc Chip Type */

/* Word definitions for board types */
#define	WD8003E		ETHERNET_MEDIA
#define	WD8003EBT	WD8003E		/* functionally identical to WD8003E */
#define	WD8003S		STARLAN_MEDIA
#define	WD8003SH	WD8003S		/* functionally identical to WD8003S */
#define	WD8003WT	TWISTED_PAIR_MEDIA
#define	WD8003W		(TWISTED_PAIR_MEDIA | INTERFACE_CHIP)
#define	WD8003EB	(ETHERNET_MEDIA | INTERFACE_CHIP)
#define	WD8003EP	WD8003EB	/* with INTERFACE_584_CHIP */
#define	WD8003EW	(EW_MEDIA | INTERFACE_CHIP)
#define	WD8013EBT	(ETHERNET_MEDIA | BOARD_16BIT)
#define	WD8013EB	(ETHERNET_MEDIA | BOARD_16BIT | INTERFACE_CHIP)
#define	WD8013W		(TWISTED_PAIR_MEDIA | BOARD_16BIT | INTERFACE_CHIP)
#define	WD8013EW	(EW_MEDIA | BOARD_16BIT | INTERFACE_CHIP)


/******************************************************************************
  This describes the smc_get_cnfg interface structure.
******************************************************************************/
typedef	struct {
  unsigned short	bid;		/* Board ID from GetBoardID	*/
  unsigned long		full_bid;	/* all Information from GetBoardID */
  unsigned short	bus;		/* 0=AT...1=MCA			*/
  unsigned short	base_io;	/* Adapter Base I/O Address	*/
  unsigned short	slot;		/* Micro Channel Slot Number	*/
  unsigned long		ram_base;	/* 32-Bit Phys Address of Shared RAM */
  unsigned short	ram_size;	/* Shared RAM Size (# of 1KB blocks) */
  unsigned short	irq_line;	/* Adapter IRQ Interrupt Line	*/
  unsigned long		rom_base;	/* 32-Bit Phys Address of Adapter ROM */
  unsigned short	rom_size;	/* Adapter ROM Size (# of 1KB blocks) */
  unsigned short	bio_new;	/* New Base I/O Address (for PutCnfg) */
  unsigned short	mode_bits1;	/* Mode bits for adapter (see below) */
  char			name[20];	/* name for board type		*/
} CNFG_Adapter;



/******************************************************************************
 This describes definitions in the smc_get_cnfg interface structure.
******************************************************************************/
/******************************************************************************
 Definitions for the field:
	cnfg_mode_bits1
******************************************************************************/
#define	INTERRUPT_STATUS_BIT	0x8000	/* PC Interrupt Line: 0 = Not Enabled */
#define	BOOT_STATUS_MASK	0x6000	/* Mask to isolate BOOT_STATUS */
#define	BOOT_INHIBIT		0x0000	/* BOOT_STATUS is 'inhibited' */
#define	BOOT_TYPE_1		0x2000	/* Unused BOOT_STATUS value */
#define	BOOT_TYPE_2		0x4000	/* Unused BOOT_STATUS value */
#define	BOOT_TYPE_3		0x6000	/* Unused BOOT_STATUS value */
#define	ZERO_WAIT_STATE_MASK	0x1800	/* Mask to isolate Wait State flags */
#define	ZERO_WAIT_STATE_8_BIT	0x1000	/* 0 = Disabled (Inserts Wait States) */
#define	ZERO_WAIT_STATE_16_BIT	0x0800	/* 0 = Disabled (Inserts Wait States) */


unsigned char irqlist[8] = {9,3,5,7,10,11,15,4};


unsigned long	GetBoardID (unsigned int, int);
int		smc_get_cnfg (CNFG_Adapter *);


#define __SLOW_DOWN_IO __asm__ __volatile__("outb %al,$0x80")

#ifdef REALLY_SLOW_IO
#define SLOW_DOWN_IO { __SLOW_DOWN_IO; __SLOW_DOWN_IO; __SLOW_DOWN_IO; __SLOW_DOWN_IO; }
#else
#define SLOW_DOWN_IO __SLOW_DOWN_IO
#endif

inline void outb(char value, unsigned short port)
{
__asm__ __volatile__ ("outb %%al,%%dx"
		::"a" ((char) value),"d" ((unsigned short) port));
}

inline unsigned int inb(unsigned short port)
{
	unsigned int _v;
__asm__ __volatile__ ("inb %%dx,%%al"
		:"=a" (_v):"d" ((unsigned short) port),"0" (0));
	return _v;
}

inline void outb_p(char value, unsigned short port)
{
__asm__ __volatile__ ("outb %%al,%%dx"
		::"a" ((char) value),"d" ((unsigned short) port));
	SLOW_DOWN_IO;
}

inline unsigned int inb_p(unsigned short port)
{
	unsigned int _v;
__asm__ __volatile__ ("inb %%dx,%%al"
		:"=a" (_v):"d" ((unsigned short) port),"0" (0));
	SLOW_DOWN_IO;
	return _v;
}



/*
	get board id information.

	returns 0 if bad board
	else returns 32 bit word with bits encoding information about
	the board. See getcnfg.h for definitions of bits.
*/

/************************************************************************
  this determines if the nic chip is a 690 or 8390.
  there are programming differences so, the driver has to know
  which chip is there.
*************************************************************************/
int is_nic_690(int ioaddr)
{
  int i,oldcr,enhval,enhval2;
  int is690 = 0;
  oldcr = inb_p(ioaddr + 0x10) & 0xfb; /* get command register, mask txp */
  outb_p((oldcr & 0x3f) | 0x80,ioaddr + 0x10); /* select page 2 */
  enhval = inb_p(ioaddr + 0x17); /* get enh */
  outb_p(enhval + 0x18,ioaddr + 0x17); /* write 0x18 to enh */
  i = inb_p(ioaddr + 0x10); /* put something else on bus */
  enhval2 = inb_p(ioaddr + 0x17) & 0x18; /* get new enh */
  if (enhval2 == 0x18) is690 = 1;
  /* restore stuff */
  outb_p(enhval,ioaddr + 0x17); /* restore enh */
  outb_p(oldcr,ioaddr + 0x10); /* restore cr */
  return(is690);
}



unsigned long smc_get_board_id(int ioaddr)
{
  int alias = 1;
  unsigned long board_id = 0;
  int board_rev,i;
  unsigned char temp1,temp2,temp7;
  /* rev number = 0 means board is broken */
  if ((board_rev = (inb_p(ioaddr+0xe) & 0x1e) >> 1) == 0) return(0);
  /* see if there is register aliasing */
  /* only check regs 1,2,3,4,7 - some ASICs don't have regs 5 and 6 */
  for (i = 1; i <= 5; i++) {
    if (inb_p(ioaddr+i) != inb_p(ioaddr+i+8)) {
      alias = 0;
      break;
    }
  }
  if (inb_p(ioaddr+7) != inb_p(ioaddr+7+8)) alias = 0;
  /* if no aliasing, check some more stuff */
  if (!alias) {
    /* see if there is an interface chip */
    temp7 = inb_p(ioaddr+7); /* save reg just in case */
    /* see if we can write and read some values in gp2 register */
    outb_p(0x35,ioaddr+7); /* write something */
    temp2 = inb_p(ioaddr + 8); /* put something else on bus */
    if (inb_p(ioaddr+7) == 0x35) {
      outb_p(0x3a,ioaddr+7); /* try another value */
      temp2 = inb_p(ioaddr + 8); /* put something else on bus */
      if (inb_p(ioaddr+7) == 0x3a) board_id |= INTERFACE_CHIP;
    }
    outb_p(temp7,ioaddr+7); /* restore reg just in case */
    /* is board 16 bit? */
    temp7 = inb_p(ioaddr+1); /* save value */
    temp1 = (temp7 & 1) ^ 1;
    outb_p(temp1,ioaddr+1); /* try to flip the 16-bit bit */
    temp2 = inb_p(ioaddr); /* put something else on bus */
    if (temp1 == (inb_p(ioaddr+1) & 1)) {
      outb_p(temp7 & 0xfe,ioaddr+1); /* restore it and clear bit 0 */
      board_id |= BOARD_16BIT;
      /* is 16 bit board in 16 bit slot? */
      if ((inb_p(ioaddr+1) & 1) == 1) board_id |= SLOT_16BIT;
    }
    else {
      outb_p(temp7,ioaddr+1); /* restore orig value */
    }
    /* find the media type */
    if ((inb_p(ioaddr+0xe) & 1) == 1) board_id |= ETHERNET_MEDIA;
    else {
      if (board_rev == 1) board_id |= STARLAN_MEDIA;
      else board_id |= TWISTED_PAIR_MEDIA;
    }
    if (board_rev >= 2) {
    /* get board id byte info */
      board_id &= STATIC_ID_MASK; /* clear out extra bits */
      temp7 = inb_p(ioaddr+0xe);
      /* check for soft config bit */
      if ((temp7 & 0x20) != 0) {
	if ((board_id == WD8003EB) || (board_id == WD8003W))
	  board_id |= ALTERNATE_IRQ_BIT;
      }
    }
    if (board_rev >= 3) {
      board_id &= BID_EEPROM_OVERRIDE;
      board_id |= INTERFACE_584_CHIP;
      /* get eeprom info */
      /* first recall the reserved engineering bytes from eeprom */
      outb_p((inb_p(ioaddr+1) & 0xc) | 2,ioaddr+1); /* set other bit */
      outb_p((inb_p(ioaddr+3) & 0xf) | 0xa0,ioaddr+3); /* set engr page */
      outb_p((inb_p(ioaddr+1) & 0xc) | 0x12,ioaddr+1); /* set rla, other bit */
      while (inb_p(ioaddr+1) & 0x10); /* wait for recall */
      if ((inb_p(ioaddr+9) & 0x18) == 8) {
	board_id |= BOARD_16BIT;
	/* is 16 bit board in 16 bit slot? */
	if ((inb_p(ioaddr+1) & 1) == 1) board_id |= SLOT_16BIT;
      }
      temp7 = inb_p(ioaddr+8);
      switch(temp7 & 7) {
      case 0: board_id |= STARLAN_MEDIA;
	break;
      case 2: board_id |= TWISTED_PAIR_MEDIA;
	break;
      case 3: board_id |= EW_MEDIA;
	break;
      default: board_id |= ETHERNET_MEDIA;
	break;
      }
      if ((temp7 & 0x18) == 8) board_id |= ALTERNATE_IRQ_BIT;
      switch(temp7 & 0xe0) {
      case 0x40: board_id |= RAM_SIZE_8K;
	break;
      case 0x60: if ((board_id & BOARD_16BIT) && !(board_id & SLOT_16BIT))
	  board_id |= RAM_SIZE_8K;
	else board_id |= RAM_SIZE_16K;
	break;
      case 0x80: board_id |= RAM_SIZE_32K;
	break;
      case 0xa0: if ((board_id & BOARD_16BIT) && !(board_id & SLOT_16BIT))
	  board_id |= RAM_SIZE_32K;
	else board_id |= RAM_SIZE_64K;
	break;
      default: board_id |= RAM_SIZE_UNKNOWN;
	break;
      }
      /* now recall the lan address from eeprom */
      outb_p((inb_p(ioaddr+1) & 0xc) | 2,ioaddr+1); /* set other bit */
      outb_p((inb_p(ioaddr+3) & 0xf) | 0x80,ioaddr+3); /* set page */
      outb_p((inb_p(ioaddr+1) & 0xc) | 0x10,ioaddr+1); /* set rla */
      while (inb_p(ioaddr+1) & 0x10); /* wait for recall */
      if (is_nic_690(ioaddr)) board_id |= NIC_690_BIT; /* check for 690 */
      return(board_id);
    }
    /* get ram size */
    if (board_rev >= 2) {
      temp1 = inb_p(ioaddr+0xe); /* get hardware id byte */
      temp2 = board_id & STATIC_ID_MASK; /* get board type */
      if ((temp2 == WD8003E) || (temp2 == WD8003S) ||
	  (temp2 == WD8003WT) || (temp2 == WD8003W) ||
	  (temp2 == WD8003EB)) {
	/* hardware ram size bit determines 8K or 32K */
	board_id |= (temp1 & 0x40) ? RAM_SIZE_32K : RAM_SIZE_8K;
      }
      else {
	if (temp2 == WD8013EBT) {
	  if (board_id & SLOT_16BIT) {
	    /* hardware ram size bit determines 16K or 64K */
	    board_id |= (temp1 & 0x40) ? RAM_SIZE_64K : RAM_SIZE_16K;
	  }
	  else {
	      /* hardware ram size bit determines 8K or 32K */
	      board_id |= (temp1 & 0x40) ? RAM_SIZE_32K : RAM_SIZE_8K;
	    }
	}
	else board_id |= RAM_SIZE_UNKNOWN;
      }
    }
    else {
      /* old rev boards */
      if (board_id & BOARD_16BIT) {
	if (board_id & SLOT_16BIT) board_id |= RAM_SIZE_16K;
	else board_id |= RAM_SIZE_8K;
      }
      else {
	if (board_id & INTERFACE_CHIP) {
	  /* look at memory size bit in register 1 */
	  board_id |= (inb_p(ioaddr+1) & 8) ? RAM_SIZE_32K : RAM_SIZE_8K;
	}
	/* can't determine ram size */
	else board_id |= RAM_SIZE_UNKNOWN;
      }
    }
    if (is_nic_690(ioaddr)) board_id |= NIC_690_BIT; /* check for 690 */
  }
 return(board_id);
}


/*
   get lots of configuration information about any western digital
   or SMC ethernet card.

   gets the following info:
   board id
   media type - starlan or ethernet or twisted pair
   interface chip type - none or 583 or 584 or 593 or 594
   NIC chip type - 8390 or 690
   8/16 bit board
   8/16 bit slot
   base address of ram
   size of ram
   interrupt line used
   base address of rom
   size of rom

   also returns the following values:
   0 = all is wonderful
   1 = no interface chip, so could not do much
   -1 = no board found
*/


int smc_get_config(CNFG_Adapter *cfg_info)
{
  unsigned long board_id, temp1;
  unsigned char csum;
  unsigned short base_addr;
  int irnum, i, rbase1;

  /* Check if board is there via checksum. */
  csum = 0;
  base_addr = cfg_info->base_io;
  for (i = 0; i < 8; i++) csum += inb_p(base_addr + 8 + i);
  if (csum != 0xff) return(-1); /* if bad checksum report no board */

  board_id = smc_get_board_id(base_addr);
  cfg_info->bid = board_id & STATIC_ID_MASK;
  cfg_info->full_bid = board_id;

  /* Copy RAM size from board_id to the config structure. */
  temp1 = board_id & RAM_SIZE_MASK; /* get RAM size bits */
  if ((temp1 >= RAM_SIZE_8K) && (temp1 <= RAM_SIZE_64K)) {
	cfg_info->ram_size = 8 << ((temp1 >> 16) - 2);
  }
  if (!(board_id & INTERFACE_CHIP)) return(1); /* no interface chip */

  /* Get interrupt line. */
  irnum = 0;
  if ((board_id & INTERFACE_CHIP_MASK) != INTERFACE_5X3_CHIP) {
	irnum = inb_p(base_addr + 1) & 4;
  }
  irnum += (inb_p(base_addr + 4) & 0x60) >> 5;
  if ((irnum == 2) && !(board_id & ALTERNATE_IRQ_BIT))
    cfg_info->irq_line = 4;
  else cfg_info->irq_line = irqlist[irnum];

  /* Get IRQ status. */

  /* Get RAM base. */
  rbase1 = inb_p(base_addr) & 0x3f;
  if ((board_id & INTERFACE_CHIP_MASK) != INTERFACE_5X3_CHIP) {
	cfg_info->ram_base =
		(inb_p(base_addr + 5) & 0x1f) << 19 | (rbase1 << 13);
  } else cfg_info->ram_base = (rbase1 | 0x40) << 13;

  /* Get ROM base. */

  /* Get ROM size. */

  /* Get boot status. */

  /* Get zero wait state. */

  /* Get name of board type. */
  switch(cfg_info->bid) {
	case WD8003W:
		strcpy(cfg_info->name,"WD8003W");
		break;
	case WD8003E:
		strcpy(cfg_info->name,"WD8003E");
		break;
	case WD8003S:
		strcpy(cfg_info->name,"WD8003S");
		break;
	case WD8003WT:
		strcpy(cfg_info->name,"WD8003WT");
		break;
	case WD8003EB:
		if ((cfg_info->full_bid & INTERFACE_CHIP_MASK) ==
			INTERFACE_584_CHIP) strcpy(cfg_info->name,"WD8003EP");
		  else strcpy(cfg_info->name,"WD8003EB");
		break;
	case WD8003EW:
		strcpy(cfg_info->name,"WD8003EW");
		break;
	case WD8013EBT:
		strcpy(cfg_info->name,"WD8013EBT");
		break;
	case WD8013EB:
		if ((cfg_info->full_bid & INTERFACE_CHIP_MASK) ==
			INTERFACE_584_CHIP) strcpy(cfg_info->name,"WD8013EP");
		  else strcpy(cfg_info->name,"WD8013EB");
   		break;
	case WD8013W:
		strcpy(cfg_info->name,"WD8013W");
		break;
	case WD8013EW:
		strcpy(cfg_info->name,"WD8013EW");
		break;
	default:
		sprintf(cfg_info->name,"%x\0",cfg_info->bid);
		break;
 }

 return(0);
}


#define PERM_OFF 0
#define PERM_ON 1
#define WD_JUMPERS 6
#define version "0.2"

unsigned char eedata[8];

void show_lan_address_regs(int ioaddr)
{
 int i,val;
 for (i=0;i < 8; i++)
 {
  printf("%02x ",eedata[i]);
 }
}

void put_back_lan_address(int ioaddr)
{
  /* put back the lan address */
  outb_p((inb_p(ioaddr+1) & 0xc) | 2,ioaddr+1); /* set other bit */
  outb_p((inb_p(ioaddr+3) & 0xf) | (8<<4),ioaddr+3); /* set page 8 */
  outb_p((inb_p(ioaddr+1) & 0xc) | 0x12,ioaddr+1); /* set rla, other bit */
  while (inb_p(ioaddr+1) & 0x10); /* wait for recall */
}

void recall_eeprom(int ioaddr, int eerow)
{
  int i;
  outb_p((inb_p(ioaddr+1) & 0xc) | 2,ioaddr+1); /* set other bit */
  outb_p((inb_p(ioaddr+3) & 0xf) | (eerow<<4),ioaddr+3); /* set page */
  outb_p((inb_p(ioaddr+1) & 0xc) | 0x12,ioaddr+1); /* set rla, other bit */
  while (inb_p(ioaddr+1) & 0x10); /* wait for recall */
  for (i = 0; i < 8; i++) {
    eedata[i] = inb_p(ioaddr + 8 + i);
  }
  put_back_lan_address(ioaddr);
}

void store_eeprom(int ioaddr, int eerow)
{
  int i;
  outb_p((inb_p(ioaddr+1) & 0xc) | 2,ioaddr+1); /* set other bit */
  outb_p((inb_p(ioaddr+3) & 0xf) | (eerow<<4),ioaddr+3); /* set page */
  outb_p((inb_p(ioaddr+1) & 0xc) | 0x82,ioaddr+1); /* set sto, other bit */
  while (inb_p(ioaddr+1) & 0x80); /* wait for store */
  put_back_lan_address(ioaddr);
}

void show_eeprom(int ioaddr)
{
 int row;
 for (row=0;row<16;row++) {
   printf("eeprom addr %02x  ",row<<3);
   recall_eeprom(ioaddr,row);
   show_lan_address_regs(ioaddr);
   printf("\n");
 }
}


void show_regs(int ioaddr)
{
  int i,page,csav;
  unsigned int reg;

  printf("ASIC regs: ");
  for (i = 0; i < 0x10; i++) {
	reg = inb_p(ioaddr + i);
	printf("%02x ",reg);
  }
  printf("\n");
  csav = inb_p(ioaddr + 0x10) & 0xc0;
  for (page = 0; page < 4; page++) {
	outb(page<<6,ioaddr + 0x10); /* set page */
	printf("NIC page %d ",page);
	for (i = 0; i < 0x10; i++) {
		reg = inb_p(ioaddr + 0x10 + i);
		printf("%02x ",reg);
	}
	printf("\n");
  }
  outb_p(csav, ioaddr + 0x10);
}


int board_there(ioaddr)
{
  int i;
  unsigned char csum;

  csum = 0;
  for (i = 0; i < 8; i++) csum += inb_p(ioaddr + 8 + i);
  if (csum != 0xff) return 0;

  return 1;
}


void main(int argc, char *argv[])
{
  unsigned char response[80];
  unsigned char boz;
  unsigned int iobase;
  int i, newval, itsok, jumpers, ee_io;
  int eestart, eeirq, eeramstart, eeramsize;
  int eerombase, eeromsize;
  unsigned short board_id;
  CNFG_Adapter smc_card_info;
  int argok, cmd_eeprom, cmd_regs, cmd_regs_verbose;
  int part1, part2, numfound;
  int found_addr[20];
  int media, eemedia;
  extern int getopt(), opterr, optind;
  extern char *optarg;

  printf("Setup for Western Digital and SMC ethercards, version %s\n", version);

  opterr = 0;
  argok = 1;
  cmd_eeprom = 0;
  cmd_regs = 0;
  cmd_regs_verbose = 0;
  while((i = getopt(argc, argv, "erv")) != EOF) switch(i) {
	case 'e':
		cmd_eeprom = 1;
		break;
	case 'r':
		cmd_regs = 1;
		break;
	case 'v':
		cmd_regs_verbose = 1;
		break;
	default:
		argok = 0;
		break;
  }

  /* Extra argument given? */
  if ((argok != 0) && (optind != argc)) {
	sscanf(argv[optind],"%x",&newval);
	if ((newval & 0xe3e0) == newval) {
		iobase = newval;
		argok = 1;
	} else {
		printf("addr must be [02468ace][0-3][02468ace]0\n");
		exit(-1);
	}
  } else iobase = 0;

  if (!argok) {
	printf("Usage: wdsetup [-e] [-r] [-v] [addr]\n");
	exit(-1);
  }

  /* Request I/O privilege. */
  if (iopl(3)) {
	perror("io-perm2");
	exit (-1);
   }

  /* Look for cards at all possible addresses. */
  if (iobase == 0) {
	numfound = 0;
	for (part1 = 0; part1 <= 7; part1++) {
		for (part2 = 0; part2 <= 0x1f; part2++) {
			iobase = (part1 << 13) + (part2 << 5);
			if (iobase > 0x100) { 
				if (board_there(iobase)) {
					/* make sure */
					if (board_there(iobase) &&
					    board_there(iobase) &&
					    board_there(iobase))
						found_addr[numfound++] = iobase;
				}
			}
		}
	}
	if (numfound == 0) {
		printf("no cards recognized.\n");
		exit(-1);
	}
	if (numfound == 1) iobase = found_addr[0];
	if (numfound > 1) {
		printf("%d cards recognized.\naddresses = ", numfound);
		for (i = 0; i < numfound; i++) {
			printf("%04x ",found_addr[i]);
		}
		printf("\nWhich address do you want to set-up ? ");
		fgets(response,80,stdin);
		sscanf(response,"%x",&iobase);
	}
  }

  smc_card_info.base_io = iobase;
  i = smc_get_config(&smc_card_info);
  if (i == -1) {
	printf("no card found\n");
	return;
  }

  if (i == 1) {
	printf("Your card does not have readable configuration information.\n");
	printf("The only way to configure it is to remove the card and set");
	printf("the jumpers. See your manual for more information.");
	return;
  }

  if (cmd_regs) {
	show_regs(iobase);
	exit(0);
  }

  if (cmd_eeprom) {
	show_eeprom(iobase);
	exit(0);
  }

  if (cmd_regs_verbose) {
	printf("Your operating system has configured the card:\n");
	printf("I/O base address\t0x%04x\n\n", iobase);
	printf("IRQ\t\t\t%d\n\n", smc_card_info.irq_line);
	printf("RAM size\t\t%d K\n\n", smc_card_info.ram_size);
	printf("RAM start\t\t%06x\n\n", smc_card_info.ram_base);
	printf("\n");
	exit(0);
  }

  /* printf("bid = 0x%08x\n", smc_card_info.full_bid); */
  printf("Board type:\t%s\n",smc_card_info.name);
  printf("node address: ");
  for (i = 0; i < 6; i++) printf("%02x",inb_p(iobase + 8 + i));
  printf("\n\n\t\t\tcurrent setup\n");
  if (!(smc_card_info.full_bid & INTERFACE_584_CHIP)) {
	printf("Adapter has a 583 ASIC, which I don't have a data sheet for. Sorry.");
	return;
  }
  jumpers = (inb_p(iobase + WD_JUMPERS) & 7) ^ 7;
  recall_eeprom(iobase,0); /* get soft config page */
  ee_io = ((eedata[2] & 0xe0) << 8) + ((eedata[2] & 0x1f) << 5);
  printf("i/o base addr\t\t%04x\n\n",ee_io);
  eeirq = 0;
  if ((smc_card_info.full_bid & INTERFACE_CHIP_MASK) != INTERFACE_5X3_CHIP) {
	eeirq = (eedata[1] & 4);
  }
  eeirq += ((eedata[4] & 0x60) >> 5);
  if ((eeirq == 2) && !(smc_card_info.full_bid & ALTERNATE_IRQ_BIT)) eeirq = 4;
    else eeirq = irqlist[eeirq];
  printf("irq\t\t\t%s%d\n\n", (eedata[4] & 0x80) ? "" : "(Disabled) ",eeirq);
  eeramsize = 8 << ((eedata[1] & 8) >> 2);
  if (smc_card_info.full_bid & BOARD_16BIT) eeramsize <<= 1;
  printf("ram size\t\t%d K\n\n",eeramsize);
  eeramstart = ((eedata[5] & 0x1f) << 19) + ((eedata[0] & 0x3f) << 13);
  printf("ram base address\t%06x\n\n",eeramstart);
  printf("add wait states\t\t%s\n\n",(eedata[4] & 1) ? "no" : "yes");
  media = smc_card_info.full_bid & MEDIA_MASK;
  eemedia = eedata[4] & 4;
  printf("network connection\t");
  switch(eemedia) {
	case 0:
		printf("AUI");
		break;
	case 4:
		printf("BNC");
		break;
	default:
		printf("unknown");
		break;
  }
  printf("\n\n");
  newval = (eedata[3] & 0xc0) >> 6;
  switch(newval) {
	case 0:
		eeromsize = 0;
		break;
	case 1:
		eeromsize = 16;
		break;
	case 2:
		eeromsize = 32;
		break;
	case 3:
		eeromsize = 64;
		break;
  }
  if (eeromsize == 0) printf("rom size\t\t(Disabled)\n\n");
    else printf("rom size\t\t%d K\n\n",eeromsize);
  eerombase = (eedata[3] & 0x3e) << 14;
  printf("rom base address\t%05x\n\n",eerombase);

  printf("change the soft configuration in eeprom? (y) ");
  fgets(response,80,stdin);
  if ((response[0] == 'y') || (response[0] == 'Y') || (response[0] == '\n')) {
	itsok = 0;
	while (!itsok) {
		printf("What io address do you want (%x) : ",ee_io);
		fgets(response,80,stdin);
		itsok = 1;
		if (response[0] != '\n') {
			sscanf(response,"%x",&newval);
			if ((newval & 0xe3e0) == newval) {
				eedata[2] = ((newval >> 5) & 0x1f) +
					     ((newval >> 8) & 0xe0);
			} else {
				itsok = 0;
				printf("addr must be [02468ace][0-3][02468ace]0\n");
			}
		}
	}

	itsok = 0;
	while (!itsok) {
		printf("What irq do you want (%d) : ",eeirq);
		fgets(response,80,stdin);
		itsok = 1;
		if (response[0] != '\n') {
			sscanf(response,"%d",&newval);
			switch(newval) {
				case 3:
					eedata[1] &= ~4;
					eedata[4] &= ~0x60;
					eedata[4] |= 0x20;
					break;
				case 4:
					eedata[1] |= 4;
					eedata[4] |= 0x60;
					break;
				case 5:
					eedata[1] &= ~4;
					eedata[4] |= 0x40;
					break;
				case 7:
					eedata[1] &= ~4;
					eedata[4] |= 0x60;
					break;
				case 9:
					eedata[1] &= ~4;
					eedata[4] &= ~0x60;
					break;
				case 10:
					eedata[1] |= 4;
					eedata[4] &= ~0x60;
					break;
				case 11:
					eedata[1] |= 4;
					eedata[4] &= ~0x60;
					eedata[4] |= 0x20;
					break;
				case 15:
					eedata[1] |= 4;
					eedata[4] &= ~0x60;
					eedata[4] |= 0x40;
					break;
				default:
					itsok = 0;
					break;
			}
		}
	}

	itsok = 0;
	while (!itsok) {
		printf("enter ram start address (%06x) : ",eeramstart);
		fgets(response,80,stdin);
		itsok = 1;
		if (response[0] != '\n') {
			sscanf(response,"%x",&newval);
			if ((newval & 0xffe000) == newval) {
				eedata[0] &= 0xc0;
				eedata[0] |= (newval & 0x07e000) >> 13;
				eedata[5] &= 0xe0;
				eedata[5] |= (newval & 0xf80000) >> 19;
			} else itsok = 0;
		}
	}

	itsok = 0;
	while (!itsok) {
		printf("add wait states ? (%s) : ",(eedata[4] & 1) ? "no" : "yes");
		fgets(response,80,stdin);
		itsok = 1;
		if (response[0] != '\n') {
			if ((response[0] == 'y') || (response[0] == 'Y')) {
				eedata[4] &= ~1;
			} else {
				if ((response[0] == 'n') || (response[0] == 'N')) {
					eedata[4] |= 1;
				} else itsok = 0;
			}
		}
	}

	itsok = 0;
	while (!itsok) {
		if ((media == ETHERNET_MEDIA) || (media == EW_MEDIA)) {
			printf("\n1 = aui");
			printf("\n2 = bnc");
		}
		if ((media == TWISTED_PAIR_MEDIA) || (media == EW_MEDIA)) {
			printf("\n3 = twisted pair");
		}
		printf("\nnetwork connection ? ");
		switch(eemedia) {
			case 0:
				printf("(AUI)");
				break;
			case 4:
				printf("(BNC)");
				break;
			default:
				printf("(unknown)");
				break;
		}
		printf(" : ");
		fgets(response,80,stdin);
		if (response[0] != '\n') {
			switch(response[0]) {
				case '1':
					if ((media == ETHERNET_MEDIA) ||
					    (media == EW_MEDIA)) {
						eedata[4] &= ~4;
						itsok = 1;
					}
					break;
				case '2':
					if ((media == ETHERNET_MEDIA) ||
					    (media == EW_MEDIA)) {
						eedata[4] |= 4;
						itsok = 1;
					}
					break;
				case '3': 
					if ((media == TWISTED_PAIR_MEDIA) ||
					    (media == EW_MEDIA)) {
						/* need info to implement this */
						itsok = 1;
					}
			}
		}
	}
   
	for (i = 0; i < 8; i++) outb_p(eedata[i],iobase + 8 + i);
	store_eeprom(iobase,0);
	printf("OK, new values stored into EEPROM 00-07\n");
	if (jumpers == 0) {
		printf("The board will configure itself from the new values");
		printf("in EEPROM\nthe next time you reboot.\n");
	} else {
		printf("Your jumpers are set to one of the hard configuration");
		printf("values.\nYou have to change the jumpers before the");
		printf("soft configuration\nwill be used by the board.\n");
	}
  }
}
